/*
<:copyright-BRCM:2002:GPL/GPL:standard

   Copyright (c) 2002 Broadcom Corporation
   All Rights Reserved

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License, version 2, as published by
the Free Software Foundation (the "GPL").

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.


A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

:>
*/
/***************************************************************************
* File Name  : bcm63xx_gpio.c
*
* Description: This file contains functions related to GPIO access.
*     See GPIO register defs in shared/broadcom/include/bcm963xx/xxx_common.h
*
*
***************************************************************************/

/*
 * Access to _SHARED_ GPIO registers should go through common functions
 * defined in board.h.  These common functions will use a spinlock with
 * irq's disabled to prevent concurrent access.
 * Functions which don't want to call the common gpio access functions must
 * acquire the bcm_gpio_spinlock with irq's disabled before accessing the
 * shared GPIO registers.
 * The GPIO registers that must be protected are:
 * GPIO->GPIODir
 * GPIO->GPIOio
 * GPIO->GPIOMode
 *
 * Note that many GPIO registers are dedicated to some driver or sub-system.
 * In those cases, the driver/sub-system can use its own locking scheme to
 * ensure serial access to its GPIO registers.
 *
 * DEFINE_SPINLOCK is the new, recommended way to declaring a spinlock.
 * See spinlock_types.h
 *
 * I am using a spin_lock_irqsave/spin_lock_irqrestore to lock and unlock
 * so that GPIO's can be accessed in interrupt context.
 */
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/smp.h>
#include <linux/bcm_assert_locks.h>
#include <bcm_map_part.h>
#include "board.h"
#include <boardparms.h>

DEFINE_SPINLOCK(bcm_gpio_spinlock);
EXPORT_SYMBOL(bcm_gpio_spinlock);


u32 gpioActiveState[GPIO_NUM_MAX];


#if defined(CONFIG_BCM96828)
/* Assign GPIO to BCM6828 PERIPH block from EPON 8051. */
#define EPON_GPIO_START             41
#define GPIO_TAKE_CONTROL(G) \
    do \
    { \
      if( (G) >= EPON_GPIO_START ) \
      { \
          GPIO->GPIOCtrl |= (GPIO_NUM_TO_MASK((G) - EPON_GPIO_START) << \
              GPIO_EPON_PERIPH_CTRL_S); \
      } \
      else if ( (G) <= 31 ) \
      { \
          GPIO->GPIOMode &= ~(GPIO_NUM_TO_MASK((G))); \
      } \
    } while(0)
#else
#define GPIO_TAKE_CONTROL(G)
#endif

#if defined(CONFIG_I2C_GPIO) && !defined(CONFIG_GPIOLIB)
/* GPIO bit functions to support drivers/i2c/busses/i2c-gpio.c */

/* noop */
int gpio_request(unsigned bpGpio, const char *label)
{
    return(0); /* success */
}

/* noop */
void gpio_free(unsigned bpGpio)
{
}

/* Assign a gpio's value. Parameter gpio is in boardparms.h format. */
void gpio_set_value(unsigned bpGpio, int value)
{
    unsigned long flags;

    /* value should be either 0 or 1 */
    spin_lock_irqsave(&bcm_gpio_spinlock, flags);
    GPIO_TAKE_CONTROL(bpGpio & BP_GPIO_NUM_MASK);
    if( value == 0 )
        GPIO->GPIOio &= ~GPIO_NUM_TO_MASK(bpGpio);
    else
        if( value == 1 )
            GPIO->GPIOio |= GPIO_NUM_TO_MASK(bpGpio);
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}

/* Set gpio direction to input. Parameter gpio is in boardparms.h format. */
int gpio_direction_input(unsigned bpGpio)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);
    GPIO_TAKE_CONTROL(bpGpio & BP_GPIO_NUM_MASK);
    GPIO->GPIODir &= ~GPIO_NUM_TO_MASK(bpGpio);
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
    return(0);
}

/* Set gpio direction to output. Parameter gpio is in boardparms.h format. */
int gpio_direction_output(unsigned bpGpio, int value)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);
    GPIO_TAKE_CONTROL(bpGpio & BP_GPIO_NUM_MASK);
    GPIO->GPIODir |= GPIO_NUM_TO_MASK(bpGpio);
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
    gpio_set_value(bpGpio, value);
    return(0);
}

/* Return a gpio's value, 0 or 1. Parameter gpio is in boardparms.h format. */
int gpio_get_value(unsigned bpGpio)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);
    GPIO_TAKE_CONTROL(bpGpio & BP_GPIO_NUM_MASK);
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
    return((int) ((GPIO->GPIOio & GPIO_NUM_TO_MASK(bpGpio)) >>
        (bpGpio & BP_GPIO_NUM_MASK)));
}
#endif

void kerSysSetGpioStateLocked(unsigned short bpGpio, GPIO_STATE_t state)
{
    BCM_ASSERT_V(bpGpio != BP_NOT_DEFINED);
    BCM_ASSERT_V((bpGpio & BP_GPIO_NUM_MASK) < GPIO_NUM_MAX);
    BCM_ASSERT_HAS_SPINLOCK_V(&bcm_gpio_spinlock);

    kerSysSetGpioDirLocked(bpGpio);

#if defined(CONFIG_BCM96828)
    GPIO_TAKE_CONTROL(bpGpio);
#endif

#if defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268)
    /* Take over high GPIOs from WLAN block */
    if ((bpGpio & BP_GPIO_NUM_MASK) > 31)
        GPIO->GPIOCtrl |= GPIO_NUM_TO_MASK(bpGpio - 32);
#endif

    if((state == kGpioActive && !(bpGpio & BP_ACTIVE_LOW)) ||
        (state == kGpioInactive && (bpGpio & BP_ACTIVE_LOW)))
        GPIO->GPIOio |= GPIO_NUM_TO_MASK(bpGpio);
    else
        GPIO->GPIOio &= ~GPIO_NUM_TO_MASK(bpGpio);

    gpioActiveState[bpGpio & BP_GPIO_NUM_MASK] = (bpGpio & BP_ACTIVE_LOW) != 0;
}

void kerSysSetGpioState(unsigned short bpGpio, GPIO_STATE_t state)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);
    kerSysSetGpioStateLocked(bpGpio, state);
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}



void kerSysSetGpioDirLocked(unsigned short bpGpio)
{
    BCM_ASSERT_V(bpGpio != BP_NOT_DEFINED);
    BCM_ASSERT_V((bpGpio & BP_GPIO_NUM_MASK) < GPIO_NUM_MAX);
    BCM_ASSERT_HAS_SPINLOCK_V(&bcm_gpio_spinlock);

    GPIO->GPIODir |= GPIO_NUM_TO_MASK(bpGpio);
}

void kerSysSetGpioDir(unsigned short bpGpio)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);

#if defined(CONFIG_BCM96828)
    GPIO_TAKE_CONTROL(bpGpio);
#endif

    kerSysSetGpioDirLocked(bpGpio);
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}

#if defined(ATI_BSP_PERSONALITY)
void kerSysSetGpioDirToOutLocked(unsigned short bpGpio)
{
    BCM_ASSERT_V(bpGpio != BP_NOT_DEFINED);
    BCM_ASSERT_V((bpGpio & BP_GPIO_NUM_MASK) < GPIO_NUM_MAX);
    BCM_ASSERT_HAS_SPINLOCK_V(&bcm_gpio_spinlock);

    GPIO->GPIODir |= GPIO_NUM_TO_MASK(bpGpio);
}

void kerSysSetGpioDirToOut(unsigned short bpGpio)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);

#if defined(CONFIG_BCM96828)
    GPIO_TAKE_CONTROL(bpGpio);
#endif

    kerSysSetGpioDirToOutLocked(bpGpio & 0x7f);
    gpioActiveState[bpGpio & 0x7f] = (bpGpio & 0x80) != 0;
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}

void kerSysSetGpioDirToInLocked(unsigned short bpGpio)
{
    BCM_ASSERT_V(bpGpio != BP_NOT_DEFINED);
    BCM_ASSERT_V((bpGpio & BP_GPIO_NUM_MASK) < GPIO_NUM_MAX);
    BCM_ASSERT_HAS_SPINLOCK_V(&bcm_gpio_spinlock);

    GPIO->GPIODir &= ~GPIO_NUM_TO_MASK(bpGpio);
}

void kerSysSetGpioDirToIn(unsigned short bpGpio)
{
    unsigned long flags;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);

#if defined(CONFIG_BCM96828)
    GPIO_TAKE_CONTROL(bpGpio);
#endif

    kerSysSetGpioDirToInLocked(bpGpio & 0x7f);
    gpioActiveState[bpGpio & 0x7f] = (bpGpio & 0x80) != 0;
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}

static int GetGpio(int gpio)
{
    return((GPIO->GPIOio & GPIO_NUM_TO_MASK(gpio))!=0);
#if 0
    if (state)
    {
        if (gpio & BP_ACTIVE_LOW)
            return(kGpioInactive);
        else
            return(kGpioActive);
    }
    else
    {
        if (gpio & BP_ACTIVE_LOW)
            return(kGpioActive);
        else
            return(kGpioInactive);
    }
#endif
}

int kerSysGetGpio(unsigned short bpGpio)
{
   if (bpGpio != BP_NOT_DEFINED)
      return(GetGpio(bpGpio));
   else
       return(-1);
}

int kerSysGetGpioState(unsigned short bpGpio)
{
  if (bpGpio != BP_NOT_DEFINED)
     return(GetGpio(bpGpio) ^ gpioActiveState[bpGpio & 0x7f] );
  else
      return(-1);
}

int kerSysGetGpioDir(unsigned short bpGpio)
{
    int dir=0;

    if (bpGpio != BP_NOT_DEFINED)
    {
        dir = (GPIO->GPIODir & GPIO_NUM_TO_MASK(bpGpio))?1:0;
        return dir;
    }
    return -1;
}

EXPORT_SYMBOL(kerSysSetGpioDirToIn);
EXPORT_SYMBOL(kerSysSetGpioDirToOut);
EXPORT_SYMBOL(kerSysGetGpio);
EXPORT_SYMBOL(kerSysGetGpioDir);
#endif

EXPORT_SYMBOL(kerSysSetGpioState);
EXPORT_SYMBOL(kerSysSetGpioStateLocked);
EXPORT_SYMBOL(kerSysSetGpioDir);
EXPORT_SYMBOL(kerSysSetGpioDirLocked);

