/*
<:copyright-gpl 
 Copyright 2002 Broadcom Corp. All Rights Reserved. 
 
 This program is free software; you can distribute it and/or modify it 
 under the terms of the GNU General Public License (Version 2) as 
 published by the Free Software Foundation. 
 
 This program is distributed in the hope it will be useful, but WITHOUT 
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
 for more details. 
 
 You should have received a copy of the GNU General Public License along 
 with this program; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place - Suite 330, Boston MA 02111-1307, USA. 
:>
*/
/***************************************************************************
 * File Name  : bcm63xx_led.c
 *
 * Description: 
 *    This file contains bcm963xx board led control API functions. 
 *
 ***************************************************************************/

/* Includes. */
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/capability.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/bcm_assert_locks.h>
#include <asm/uaccess.h>

#include <bcm_map_part.h>
#include <board.h>
#include <boardparms.h>

extern spinlock_t bcm_gpio_spinlock;
#if defined(ATI_BSP_PERSONALITY)
  #include <ati_hwdefs.h>

extern ATI_NVDATA atiNvramData;
unsigned char atiLeds[MAX_LED_COUNT][256];
#endif

#define k125ms              (HZ / 8)   // 125 ms
#define kFastBlinkCount     0          // 125ms
#define kSlowBlinkCount     1          // 250ms

#define kLedOff             0
#define kLedOn              1

#define kLedGreen           0
#define kLedRed             1

// uncomment // for debug led
// #define DEBUG_LED

typedef int (*BP_LED_FUNC) (unsigned short *);

typedef struct {
    BOARD_LED_NAME ledName;
    BP_LED_FUNC bpFunc;
    BP_LED_FUNC bpFuncFail;
} BP_LED_INFO, *PBP_LED_INFO;

typedef struct {
    short ledGreenGpio;             // GPIO # for LED
    short ledRedGpio;               // GPIO # for Fail LED
    BOARD_LED_STATE ledState;       // current led state
    short blinkCountDown;           // Count for blink states
#if defined(ATI_BSP_PERSONALITY)
  char name[256];
#endif
} LED_CTRL, *PLED_CTRL;

#if !defined(ATI_BSP_PERSONALITY)
static BP_LED_INFO bpLedInfo[] =
#else
static BP_LED_INFO bpLedInfo[kLedEnd] =
#endif
{
#if !defined(ATI_BSP_PERSONALITY)
  {kLedAdsl, BpGetAdslLedGpio, BpGetAdslFailLedGpio},
  {kLedSecAdsl, BpGetSecAdslLedGpio, BpGetSecAdslFailLedGpio},
  {kLedWanData, BpGetWanDataLedGpio, BpGetWanErrorLedGpio},
  {kLedSes, BpGetWirelessSesLedGpio, NULL},
  {kLedVoip, BpGetVoipLedGpio, NULL},
  {kLedVoip1, BpGetVoip1LedGpio, BpGetVoip1FailLedGpio},
  {kLedVoip2, BpGetVoip2LedGpio, BpGetVoip2FailLedGpio},
  {kLedPots, BpGetPotsLedGpio, NULL},
  {kLedDect, BpGetDectLedGpio, NULL},
  {kLedGpon, BpGetGponLedGpio, BpGetGponFailLedGpio},
  {kLedMoCA, BpGetMoCALedGpio, BpGetMoCAFailLedGpio},
#endif
  {kLedEnd, NULL, NULL}
};

// global variables:
static struct timer_list gLedTimer;
static PLED_CTRL gLedCtrl = NULL;
static int gTimerOn = FALSE;
static int gTimerOnRequests = 0;
static unsigned int gLedRunningCounter = 0;  // only used by WLAN

void (*ethsw_led_control)(unsigned long ledMask, int value) = NULL;
EXPORT_SYMBOL(ethsw_led_control);

/** This spinlock protects all access to gLedCtrl, gTimerOn
 *  gTimerOnRequests, gLedRunningCounter, and ledTimerStart function.
 *  Use spin_lock_irqsave to lock the spinlock because ledTimerStart
 *  may be called from interrupt handler (WLAN?)
 */
static spinlock_t brcm_ledlock;
static void ledTimerExpire(void);

//**************************************************************************************
// LED operations
//**************************************************************************************

#if defined(ATI_BSP_PERSONALITY)
int getLed(BOARD_LED_NAME led, BOARD_LED_STATE *led_state, char **name, int *greenGpio, int *redGpio, int *blinkCnt)
{

  if (led < kLedEnd)
  {
    if (gLedCtrl && (gLedCtrl[led].ledGreenGpio != -1 || gLedCtrl[led].ledRedGpio != -1))
    {
      *name      = gLedCtrl[led].name;
      *led_state = gLedCtrl[led].ledState;
      *greenGpio = gLedCtrl[led].ledGreenGpio;
      *redGpio   = gLedCtrl[led].ledRedGpio;
      *blinkCnt  = gLedCtrl[led].blinkCountDown;
    }
    else
      return(1);
    return(0);
  }
  return(1);
}
#endif

// turn led on and set the ledState
static void setLed (PLED_CTRL pLed, unsigned short led_state, unsigned short led_type)
{
    short led_gpio;
    unsigned short gpio_state;
    unsigned long flags;

    if (led_type == kLedRed)
        led_gpio = pLed->ledRedGpio;
    else
        led_gpio = pLed->ledGreenGpio;

    if (led_gpio == -1)
        return;

    if (((led_gpio & BP_ACTIVE_LOW) && (led_state == kLedOn)) || 
        (!(led_gpio & BP_ACTIVE_LOW) && (led_state == kLedOff)))
        gpio_state = 0;
    else
        gpio_state = 1;

    /* spinlock to protect access to GPIODir, GPIOio */
    spin_lock_irqsave(&bcm_gpio_spinlock, flags);

#if defined(CONFIG_BCM96328)
    /* Enable LED controller to drive this GPIO */
    if (!(led_gpio & BP_GPIO_SERIAL))
        GPIO->GPIOMode |= GPIO_NUM_TO_MASK(led_gpio);
#endif

#if defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318)|| defined(CONFIG_BCM96828)
    /* Enable LED controller to drive this GPIO */
    if ((led_gpio & BP_GPIO_NUM_MASK) < 24 &&  !(led_gpio & BP_GPIO_SERIAL))
        GPIO->LEDCtrl |= GPIO_NUM_TO_MASK(led_gpio);
#endif

#if defined(CONFIG_BCM96328) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318)|| defined(CONFIG_BCM96828)
    if ((led_gpio & BP_GPIO_NUM_MASK) < 24)
    {
      LED->ledMode &= ~(LED_MODE_MASK << GPIO_NUM_TO_LED_MODE_SHIFT(led_gpio));
      if( gpio_state )
          LED->ledMode |= (LED_MODE_OFF << GPIO_NUM_TO_LED_MODE_SHIFT(led_gpio));
      else
          LED->ledMode |= (LED_MODE_ON << GPIO_NUM_TO_LED_MODE_SHIFT(led_gpio));
    }
    else 
    {
        GPIO->GPIODir |= GPIO_NUM_TO_MASK(led_gpio);
        if( gpio_state )
            GPIO->GPIOio |= GPIO_NUM_TO_MASK(led_gpio);
        else
            GPIO->GPIOio &= ~GPIO_NUM_TO_MASK(led_gpio);
    }

#else
    if (led_gpio & BP_GPIO_SERIAL) {
        while (GPIO->SerialLedCtrl & SER_LED_BUSY);
        if( gpio_state )
            GPIO->SerialLed |= GPIO_NUM_TO_MASK(led_gpio);
        else
            GPIO->SerialLed &= ~GPIO_NUM_TO_MASK(led_gpio);
    }
    else 
    {
        GPIO->GPIODir |= GPIO_NUM_TO_MASK(led_gpio);
        if( gpio_state )
            GPIO->GPIOio |= GPIO_NUM_TO_MASK(led_gpio);
        else
            GPIO->GPIOio &= ~GPIO_NUM_TO_MASK(led_gpio);
    }
#endif

    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}

// toggle the LED
static void ledToggle(PLED_CTRL pLed)
{
    short led_gpio;
    unsigned long flags;

    led_gpio = pLed->ledGreenGpio;

    if (led_gpio == -1)
        return;

    spin_lock_irqsave(&bcm_gpio_spinlock, flags);
#if defined(CONFIG_BCM96328) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318)|| defined(CONFIG_BCM96828)
    if ((led_gpio & BP_GPIO_NUM_MASK) < 24)
    {
      LED->ledMode ^= (LED_MODE_MASK << GPIO_NUM_TO_LED_MODE_SHIFT(led_gpio));
    }
    else {
        GPIO->GPIODir |= GPIO_NUM_TO_MASK(led_gpio);
        GPIO->GPIOio ^= GPIO_NUM_TO_MASK(led_gpio);
    }
#else

    if (led_gpio & BP_GPIO_SERIAL) {
        while (GPIO->SerialLedCtrl & SER_LED_BUSY);
        GPIO->SerialLed ^= GPIO_NUM_TO_MASK(led_gpio);
    }
    else {
        GPIO->GPIODir |= GPIO_NUM_TO_MASK(led_gpio);
        GPIO->GPIOio ^= GPIO_NUM_TO_MASK(led_gpio);
    }

#endif
    spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}   

static void ledToggleFail(PLED_CTRL pLed)
{
  short led_gpio;
  unsigned long flags;

  led_gpio = pLed->ledRedGpio;

  if (led_gpio == -1)
    return;

  spin_lock_irqsave(&bcm_gpio_spinlock, flags);
#if defined(CONFIG_BCM96328) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318)|| defined(CONFIG_BCM96828)
  if ((led_gpio & BP_GPIO_NUM_MASK) < 24)
  {
    LED->ledMode ^= (LED_MODE_MASK << GPIO_NUM_TO_LED_MODE_SHIFT(led_gpio));
  }
  else
  {
    GPIO->GPIODir |= GPIO_NUM_TO_MASK(led_gpio);
    GPIO->GPIOio ^= GPIO_NUM_TO_MASK(led_gpio);
  }
#else

  if (led_gpio & BP_GPIO_SERIAL)
  {
    while (GPIO->SerialLedCtrl & SER_LED_BUSY);
    GPIO->SerialLed ^= GPIO_NUM_TO_MASK(led_gpio);
  }
  else
  {
//printk("R%d%c ", led_gpio&0x7f, (GPIO->GPIOio & GPIO_NUM_TO_MASK(led_gpio))?'-':'+');
    GPIO->GPIODir |= GPIO_NUM_TO_MASK(led_gpio);
    GPIO->GPIOio ^= GPIO_NUM_TO_MASK(led_gpio);
  }

#endif
  spin_unlock_irqrestore(&bcm_gpio_spinlock, flags);
}   

static void ledToggleBoth(PLED_CTRL pLed)
{
  ledToggle(pLed);
  ledToggleFail(pLed);
}


/** Start the LED timer
 *
 * Must be called with brcm_ledlock held
 */
static void ledTimerStart(void)
{
#if defined(DEBUG_LED)
    printk("led: add_timer\n");
#endif

    BCM_ASSERT_HAS_SPINLOCK_C(&brcm_ledlock);

    init_timer(&gLedTimer);
    gLedTimer.function = (void*)ledTimerExpire;
    gLedTimer.expires = jiffies + k125ms;        // timer expires in ~100ms
    add_timer (&gLedTimer);
} 


// led timer expire kicks in about ~100ms and perform the led operation according to the ledState and
// restart the timer according to ledState
static void ledTimerExpire(void)
{
    int i;
    PLED_CTRL pCurLed;
    unsigned long flags;

    BCM_ASSERT_NOT_HAS_SPINLOCK_V(&brcm_ledlock);

    for (i = 0, pCurLed = gLedCtrl; i < kLedEnd; i++, pCurLed++)
    {
#if defined(DEBUG_LED)
        printk("led[%d]: Mask=0x%04x, State = %d, blcd=%d\n", i, pCurLed->ledGreenGpio, pCurLed->ledState, pCurLed->blinkCountDown);
#endif
        spin_lock_irqsave(&brcm_ledlock, flags);        // LEDs can be changed from ISR
        switch (pCurLed->ledState)
        {
        case kLedStateOn:
        case kLedStateOff:
        case kAtiFailLedStateOn:
        case kLedStateFail:
            pCurLed->blinkCountDown = 0;            // reset the blink count down
            spin_unlock_irqrestore(&brcm_ledlock, flags);
            break;

        case kLedStateSlowBlinkContinues:
            if (pCurLed->blinkCountDown-- == 0)
            {
                pCurLed->blinkCountDown = kSlowBlinkCount;
                ledToggle(pCurLed);
            }
            gTimerOnRequests++;
            spin_unlock_irqrestore(&brcm_ledlock, flags);
            break;

        case kLedStateFastBlinkContinues:
            if (pCurLed->blinkCountDown-- == 0)
            {
                pCurLed->blinkCountDown = kFastBlinkCount;
                ledToggle(pCurLed);
            }
            gTimerOnRequests++;
            spin_unlock_irqrestore(&brcm_ledlock, flags);
            break;

        case kLedStateUserWpsInProgress:          /* 200ms on, 100ms off */
            if (pCurLed->blinkCountDown-- == 0)
            {
                pCurLed->blinkCountDown = (((gLedRunningCounter++)&1)? kFastBlinkCount: kSlowBlinkCount);
                ledToggle(pCurLed);
            }
            gTimerOnRequests++;
            spin_unlock_irqrestore(&brcm_ledlock, flags);
            break;             

        case kLedStateUserWpsError:               /* 100ms on, 100ms off */
            if (pCurLed->blinkCountDown-- == 0)
            {
                pCurLed->blinkCountDown = kFastBlinkCount;
                ledToggle(pCurLed);
            }
            gTimerOnRequests++;
            spin_unlock_irqrestore(&brcm_ledlock, flags);
            break;        

        case kLedStateUserWpsSessionOverLap:      /* 100ms on, 100ms off, 5 times, off for 500ms */        
            if (pCurLed->blinkCountDown-- == 0)
            {
                if(((++gLedRunningCounter)%10) == 0) {
                    pCurLed->blinkCountDown = 4;
                    setLed(pCurLed, kLedOff, kLedGreen);
                    pCurLed->ledState = kLedStateUserWpsSessionOverLap;
                }
                else
                {
                    pCurLed->blinkCountDown = kFastBlinkCount;
                    ledToggle(pCurLed);
                }
            }
            gTimerOnRequests++;
            spin_unlock_irqrestore(&brcm_ledlock, flags);
            break;        

#if defined(ATI_BSP_PERSONALITY)
//Fail
    case kAtiFailLedStateSlowBlinkContinues:
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = kSlowBlinkCount;
        ledToggleFail(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;

    case kAtiFailLedStateFastBlinkContinues:
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = kFastBlinkCount;
        ledToggleFail(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;

    case kAtiFailLedStateUserWpsInProgress:          /* 200ms on, 100ms off */
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = (((gLedRunningCounter++)&1)? kFastBlinkCount: kSlowBlinkCount);
        ledToggleFail(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;             

    case kAtiFailLedStateUserWpsError:               /* 100ms on, 100ms off */
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = kFastBlinkCount;
        ledToggleFail(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;        

    case kAtiFailLedStateUserWpsSessionOverLap:      /* 100ms on, 100ms off, 5 times, off for 500ms */        
      if (pCurLed->blinkCountDown-- == 0)
      {
        if (((++gLedRunningCounter)%10) == 0)
        {
          pCurLed->blinkCountDown = 4;
          setLed(pCurLed, kLedOff, kLedRed);
          pCurLed->ledState = kAtiFailLedStateUserWpsSessionOverLap;
        }
        else
        {
          pCurLed->blinkCountDown = kFastBlinkCount;
          ledToggleFail(pCurLed);
        }
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;        
//Both
    case kAtiBothLedStateSlowBlinkContinues:
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = kSlowBlinkCount;
        ledToggleBoth(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;

    case kAtiBothLedStateFastBlinkContinues:
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = kFastBlinkCount;
        ledToggleBoth(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;

    case kAtiBothLedStateUserWpsInProgress:          /* 200ms on, 100ms off */
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = (((gLedRunningCounter++)&1)? kFastBlinkCount: kSlowBlinkCount);
        ledToggleBoth(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;             

    case kAtiBothLedStateUserWpsError:               /* 100ms on, 100ms off */
      if (pCurLed->blinkCountDown-- == 0)
      {
        pCurLed->blinkCountDown = kFastBlinkCount;
        ledToggleBoth(pCurLed);
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;        

    case kAtiBothLedStateUserWpsSessionOverLap:      /* 100ms on, 100ms off, 5 times, off for 500ms */        
      if (pCurLed->blinkCountDown-- == 0)
      {
        if (((++gLedRunningCounter)%10) == 0)
        {
          pCurLed->blinkCountDown = 4;
          setLed(pCurLed, kLedOff, kLedGreen);
          setLed(pCurLed, kLedOff, kLedRed);
          pCurLed->ledState = kAtiFailLedStateUserWpsSessionOverLap;
        }
        else
        {
          pCurLed->blinkCountDown = kFastBlinkCount;
          ledToggleBoth(pCurLed);
        }
      }
      gTimerOnRequests++;                 
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      break;        

#endif
    default:
      spin_unlock_irqrestore(&brcm_ledlock, flags);
      printk("Invalid led state = %d\n", pCurLed->ledState);
    }
  }

    // Restart the timer if any of our previous LED operations required
    // us to restart the timer, or if any other threads requested a timer
    // restart.  Note that throughout this function, gTimerOn == TRUE, so
    // any other thread which want to restart the timer would only
    // increment the gTimerOnRequests and not call ledTimerStart.
    spin_lock_irqsave(&brcm_ledlock, flags);
    if (gTimerOnRequests > 0)
    {
        ledTimerStart();
        gTimerOnRequests = 0;
    }
    else
    {
        gTimerOn = FALSE;
    }
    spin_unlock_irqrestore(&brcm_ledlock, flags);
}

void __init boardLedInit(void)
{
    PBP_LED_INFO pInfo;
    unsigned short i;
    short gpio;
    ETHERNET_MAC_INFO EnetInfo[BP_MAX_ENET_MACS];

    spin_lock_init(&brcm_ledlock);

#if !(defined(CONFIG_BCM96328) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318)|| defined(CONFIG_BCM96828))
    /* Set blink rate for hardware LEDs. */
    GPIO->LEDCtrl &= ~LED_INTERVAL_SET_MASK;
    GPIO->LEDCtrl |= LED_INTERVAL_SET_80MS;
#endif

    gLedCtrl = (PLED_CTRL) kmalloc((kLedEnd * sizeof(LED_CTRL)), GFP_KERNEL);

    if( gLedCtrl == NULL ) {
        printk( "LED memory allocation error.\n" );
        return;
    }

    /* Initialize LED control */
    for (i = 0; i < kLedEnd; i++) {
        gLedCtrl[i].ledGreenGpio = -1;
        gLedCtrl[i].ledRedGpio = -1;
        gLedCtrl[i].ledState = kLedStateOff;
        gLedCtrl[i].blinkCountDown = 0;            // reset the blink count down
#if defined(ATI_BSP_PERSONALITY)
    gLedCtrl[i].name[0]='\0';
#endif
    }

    for( pInfo = bpLedInfo; pInfo->ledName != kLedEnd; pInfo++ ) {
        if( pInfo->bpFunc && (*pInfo->bpFunc) (&gpio) == BP_SUCCESS )
        {
            gLedCtrl[pInfo->ledName].ledGreenGpio = gpio;
        }
        if( pInfo->bpFuncFail && (*pInfo->bpFuncFail)(&gpio)==BP_SUCCESS )
        {
            gLedCtrl[pInfo->ledName].ledRedGpio = gpio;
        }
        setLed(&gLedCtrl[pInfo->ledName], kLedOff, kLedGreen);
        setLed(&gLedCtrl[pInfo->ledName], kLedOff, kLedRed);
    }

#if defined(ATI_BSP_PERSONALITY)
  {
    int foundExSwitchReset = 0;
    for (i=0; i < 10 && atiLeds[i][0]!=0xff; i++) //LED ID
    {
      snprintf(gLedCtrl[atiLeds[i][0]].name, sizeof(gLedCtrl[i].name), "%s", &atiLeds[i][3]);
      if (atiLeds[i][1]!=0xff) // Green
      {
        gLedCtrl[atiLeds[i][0]].ledGreenGpio = 
        (atiLeds[i][1] & 0x007f) | ((atiLeds[i][1] & 0x0080) << 8); //convert to BP_ACTIVE_LOW
      }
      if (atiLeds[i][2]!=0xff) // Red
      {
        gLedCtrl[atiLeds[i][0]].ledRedGpio = 
        (atiLeds[i][2] & 0x007f) | ((atiLeds[i][2] & 0x0080) << 8); //convert to BP_ACTIVE_LOW
      }
      setLed(&gLedCtrl[atiLeds[i][0]], kLedOff, kLedGreen);
      if (kLedAtiSystem == atiLeds[i][0])
        setLed(&gLedCtrl[atiLeds[i][0]], kLedOn, kLedRed);
      else
        setLed(&gLedCtrl[atiLeds[i][0]], kLedOff, kLedRed);
      if (!strcmp(&atiLeds[i][3], "AtiResetExSwitch"))
      {
        foundExSwitchReset = 1;
      }
    }

  {
    if(BpGetEthernetMacInfo(&EnetInfo[0], BP_MAX_ENET_MACS) == BP_SUCCESS)
    {
       if (BP_ENET_EXTERNAL_SWITCH != EnetInfo[1].ucPhyType) // No second switch
         foundExSwitchReset = 1; // Don't need to create
    }
    else
      foundExSwitchReset = 1; // Don't create if an error occured.
 

    if (!foundExSwitchReset)
    {
      snprintf(gLedCtrl[kResetAtiSwitch].name, sizeof(gLedCtrl[kResetAtiSwitch].name), "%s", "AtiResetExSwitch");
      gLedCtrl[kResetAtiSwitch].ledGreenGpio = 0x0d;
      gLedCtrl[kResetAtiSwitch].ledRedGpio = 0xff;
      setLed(&gLedCtrl[kResetAtiSwitch], kLedOff, kLedGreen);
    }
  }
}
#endif

#if defined(CONFIG_BCM96816) || defined(CONFIG_BCM96818)
  if ( BpGetEthernetMacInfo( &EnetInfo[0], 1 ) == BP_SUCCESS )
  {
    if ( EnetInfo[0].sw.ledInfo[0].duplexLed != BP_NOT_DEFINED )
    {
#if defined(ATI_BSP_PERSONALITY)
      snprintf(gLedCtrl[kLedEth0Duplex].name, sizeof(gLedCtrl[i].name), "Eth0Duplex");
#endif
      gLedCtrl[kLedEth0Duplex].ledGreenGpio = EnetInfo[0].sw.ledInfo[0].duplexLed;
      setLed(&gLedCtrl[kLedEth0Duplex], kLedOff, kLedGreen);
    }
    if ( EnetInfo[0].sw.ledInfo[0].speedLed100 != BP_NOT_DEFINED )
    {
#if defined(ATI_BSP_PERSONALITY)
      snprintf(gLedCtrl[kLedEth0Spd100].name, sizeof(gLedCtrl[i].name), "Eth0Spd100");
#endif
      gLedCtrl[kLedEth0Spd100].ledGreenGpio = EnetInfo[0].sw.ledInfo[0].speedLed100;
      setLed(&gLedCtrl[kLedEth0Spd100], kLedOff, kLedGreen);
    }
    if ( EnetInfo[0].sw.ledInfo[0].speedLed1000 != BP_NOT_DEFINED )
    {
#if defined(ATI_BSP_PERSONALITY)
      snprintf(gLedCtrl[kLedEth0Spd1000].name, sizeof(gLedCtrl[i].name), "Eth0Spd1000");
#endif
      gLedCtrl[kLedEth0Spd1000].ledGreenGpio = EnetInfo[0].sw.ledInfo[0].speedLed1000;
      setLed(&gLedCtrl[kLedEth0Spd1000], kLedOff, kLedGreen);
    }

    if ( EnetInfo[0].sw.ledInfo[1].duplexLed != BP_NOT_DEFINED )
    {
#if defined(ATI_BSP_PERSONALITY)
      snprintf(gLedCtrl[kLedEth1Duplex].name, sizeof(gLedCtrl[i].name), "Eth1Duplex");
#endif
      gLedCtrl[kLedEth1Duplex].ledGreenGpio = EnetInfo[0].sw.ledInfo[1].duplexLed;
      setLed(&gLedCtrl[kLedEth1Duplex], kLedOff, kLedGreen);
    }
    if ( EnetInfo[0].sw.ledInfo[1].speedLed100 != BP_NOT_DEFINED )
    {
#if defined(ATI_BSP_PERSONALITY)
      snprintf(gLedCtrl[kLedEth1Spd100].name, sizeof(gLedCtrl[i].name), "Eth1Spd100");
#endif
      gLedCtrl[kLedEth1Spd100].ledGreenGpio = EnetInfo[0].sw.ledInfo[1].speedLed100;
      setLed(&gLedCtrl[kLedEth1Spd100], kLedOff, kLedGreen);
    }
    if ( EnetInfo[0].sw.ledInfo[1].speedLed1000 != BP_NOT_DEFINED )
    {
#if defined(ATI_BSP_PERSONALITY)
      snprintf(gLedCtrl[kLedEth1Spd1000].name, sizeof(gLedCtrl[i].name), "Eth1Spd1000");
#endif
      gLedCtrl[kLedEth1Spd1000].ledGreenGpio = EnetInfo[0].sw.ledInfo[1].speedLed1000;
      setLed(&gLedCtrl[kLedEth1Spd1000], kLedOff, kLedGreen);
    }
  }
#endif

#if defined(CONFIG_BCM96818)
      /*
       * The following GPIO may be tri-stated due to MII_OVER_GPIO feature, so make sure they 
       * are not. GPIO[30:33], GPIO[39]. 
       */
  {
      int i;
      for (i = 0; i < kLedEnd; i++)
      {
          short greenGpio = gLedCtrl[i].ledGreenGpio & 0x007f;
          short redGpio   = gLedCtrl[i].ledRedGpio & 0x007f;

          if (greenGpio == 30 || greenGpio == 31 || greenGpio == 32 || greenGpio == 33 || greenGpio == 39 ||
              redGpio == 30   || redGpio == 31   || redGpio == 32   || redGpio == 33   || redGpio == 39)
          {
              GPIO->MiiPadCtrl &= ~0xFFFFFF00;
              GPIO->MiiPadCtrl |= 0xD0000000;
              break;
          }
      }
  }
#endif


#if defined(DEBUG_LED)
    for (i=0; i < kLedEnd; i++)
        printk("initLed: led[%d]: Gpio=0x%04x, FailGpio=0x%04x\n", i, gLedCtrl[i].ledGreenGpio, gLedCtrl[i].ledRedGpio);
#endif
}

// led ctrl.  Maps the ledName to the corresponding ledInfoPtr and perform the led operation
void boardLedCtrl(BOARD_LED_NAME ledName, BOARD_LED_STATE ledState)
{
    unsigned long flags;
    PLED_CTRL pLed;

    BCM_ASSERT_NOT_HAS_SPINLOCK_V(&brcm_ledlock);

    spin_lock_irqsave(&brcm_ledlock, flags);     // LED can be changed from ISR

    if( (int) ledName < kLedEnd ) {

        pLed = &gLedCtrl[ledName];

        // If the state is kLedStateFail and there is not a failure LED defined
        // in the board parameters, change the state to kLedStateSlowBlinkContinues.
        if( ledState == kLedStateFail && pLed->ledRedGpio == -1 )
            ledState = kLedStateSlowBlinkContinues;

        // Save current LED state
        pLed->ledState = ledState;

        // Start from LED Off and turn it on later as needed
        setLed (pLed, kLedOff, kLedGreen);
        setLed (pLed, kLedOff, kLedRed);

        // Disable HW control for WAN Data LED. 
        // It will be enabled only if LED state is On
#if defined(CONFIG_BCM96328) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318)|| defined(CONFIG_BCM96828)
        if (ledName == kLedWanData)
            LED->ledHWDis |= GPIO_NUM_TO_MASK(pLed->ledGreenGpio);
#elif defined(CONFIG_BCM96368)
        if (ledName == kLedWanData) {
            GPIO->AuxLedCtrl |= AUX_HW_DISAB_2;
            if (pLed->ledGreenGpio & BP_ACTIVE_LOW)
                GPIO->AuxLedCtrl |= (LED_STEADY_ON << AUX_MODE_SHFT_2);
            else
                GPIO->AuxLedCtrl &= ~(LED_STEADY_ON << AUX_MODE_SHFT_2);
        }
#endif

        switch (ledState) {
        case kLedStateOn:
            // Enable SAR to control INET LED
#if defined(CONFIG_BCM96328) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318)|| defined(CONFIG_BCM96828)
            if (ledName == kLedWanData)
                LED->ledHWDis &= ~GPIO_NUM_TO_MASK(pLed->ledGreenGpio);
#elif defined(CONFIG_BCM96368)
            if (ledName == kLedWanData) {
                GPIO->AuxLedCtrl &= ~AUX_HW_DISAB_2;
                if (pLed->ledGreenGpio & BP_ACTIVE_LOW)
                    GPIO->AuxLedCtrl &= ~(LED_STEADY_ON << AUX_MODE_SHFT_2);
                else
                    GPIO->AuxLedCtrl |= (LED_STEADY_ON << AUX_MODE_SHFT_2);
            }
#endif
            setLed (pLed, kLedOn, kLedGreen);
            break;

        case kLedStateOff:
            break;

        case kLedStateFail:
            setLed (pLed, kLedOn, kLedRed);
            break;

        case kLedStateSlowBlinkContinues:
            pLed->blinkCountDown = kSlowBlinkCount;
            gTimerOnRequests++;
            break;

        case kLedStateFastBlinkContinues:
            pLed->blinkCountDown = kFastBlinkCount;
            gTimerOnRequests++;
            break;

        case kLedStateUserWpsInProgress:          /* 200ms on, 100ms off */
            pLed->blinkCountDown = kFastBlinkCount;
            gLedRunningCounter = 0;
            gTimerOnRequests++;
            break;             

        case kLedStateUserWpsError:               /* 100ms on, 100ms off */
            pLed->blinkCountDown = kFastBlinkCount;
            gLedRunningCounter = 0;
            gTimerOnRequests++;
            break;        

        case kLedStateUserWpsSessionOverLap:      /* 100ms on, 100ms off, 5 times, off for 500ms */
            pLed->blinkCountDown = kFastBlinkCount;
            gTimerOnRequests++;
            break;        

#if defined(ATI_BSP_PERSONALITY)
//Fail
    case kAtiFailLedStateOn:
      setLed (pLed, kLedOn, kLedRed);
      break;

    case kAtiFailLedStateSlowBlinkContinues:
      pLed->blinkCountDown = kSlowBlinkCount;
      gTimerOnRequests++;
      break;

    case kAtiFailLedStateFastBlinkContinues:
      pLed->blinkCountDown = kFastBlinkCount;
      gTimerOnRequests++;
      break;

    case kAtiFailLedStateUserWpsInProgress:          /* 200ms on, 100ms off */
      pLed->blinkCountDown = kFastBlinkCount;
      gLedRunningCounter = 0;
      gTimerOnRequests++;
      break;             

    case kAtiFailLedStateUserWpsError:               /* 100ms on, 100ms off */
      pLed->blinkCountDown = kFastBlinkCount;
      gLedRunningCounter = 0;
      gTimerOnRequests++;
      break;        

    case kAtiFailLedStateUserWpsSessionOverLap:      /* 100ms on, 100ms off, 5 times, off for 500ms */
      pLed->blinkCountDown = kFastBlinkCount;
      gTimerOnRequests++;
      break;        

//Both
    case kAtiBothLedStateOn:
      setLed (pLed, kLedOn, kLedGreen);
      setLed (pLed, kLedOn, kLedRed);
      break;

    case kAtiBothLedStateSlowBlinkContinues:
      pLed->blinkCountDown = kSlowBlinkCount;
      gTimerOnRequests++;
      break;

    case kAtiBothLedStateFastBlinkContinues:
      pLed->blinkCountDown = kFastBlinkCount;
      gTimerOnRequests++;
      break;

    case kAtiBothLedStateUserWpsInProgress:          /* 200ms on, 100ms off */
      pLed->blinkCountDown = kFastBlinkCount;
      gLedRunningCounter = 0;
      gTimerOnRequests++;
      break;             

    case kAtiBothLedStateUserWpsError:               /* 100ms on, 100ms off */
      pLed->blinkCountDown = kFastBlinkCount;
      gLedRunningCounter = 0;
      gTimerOnRequests++;

    case kAtiBothLedStateUserWpsSessionOverLap:      /* 100ms on, 100ms off, 5 times, off for 500ms */
      pLed->blinkCountDown = kFastBlinkCount;
      gTimerOnRequests++;
      break;        
#endif

    default:
      printk("Invalid led state = %d\n", ledState);
    }
  }

    // If gTimerOn is false, that means ledTimerExpire has already finished
    // running and has not restarted the timer.  Then we can start it here.
    // Otherwise, we only leave the gTimerOnRequests > 0 which will be
    // noticed at the end of the ledTimerExpire function.
    if (!gTimerOn && gTimerOnRequests > 0)
    {
        ledTimerStart();
        gTimerOn = TRUE;
        gTimerOnRequests = 0;
    }
    spin_unlock_irqrestore(&brcm_ledlock, flags);
}
