/*
<:copyright-BRCM:2002:GPL/GPL:standard

   Copyright (c) 2002 Broadcom Corporation
   All Rights Reserved

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License, version 2, as published by
the Free Software Foundation (the "GPL").

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.


A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

:>
*/
/***************************************************************************
* File Name  : board.c
*
* Description: This file contains Linux character device driver entry
*              for the board related ioctl calls: flash, get free kernel
*              page and dump kernel memory, etc.
*
*
***************************************************************************/

/* Includes. */
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/capability.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <linux/pagemap.h>
#include <asm/uaccess.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/if.h>
#include <linux/pci.h>
#include <linux/ctype.h>
#include <linux/proc_fs.h>
#include <linux/smp.h>
#include <linux/version.h>
#include <linux/reboot.h>
#include <linux/bcm_assert_locks.h>
#include <asm/delay.h>

#include <bcmnetlink.h>
#include <net/sock.h>
#include <bcm_map_part.h>
#include <board.h>
#include <spidevices.h>
#define  BCMTAG_EXE_USE
#include <bcmTag.h>
#include <boardparms.h>
#include <boardparms_voice.h>
#include <flash_api.h>
#include <bcm_intr.h>
#include <flash_common.h>
#include <shared_utils.h>
#include <bcmpci.h>
#include <linux/bcm_log.h>
#include <bcmSpiRes.h>
#if defined(ATI_PRODUCT_CONFIG) && (defined(CONFIG_ATI_SFP) || defined(CONFIG_BCM_I2C_BUS))
#if !defined(CONFIG_BCM_I2C_BUS)
#include <i2c-ati-gpio.h>
#endif
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#endif
//extern unsigned int flash_get_reserved_bytes_at_end(const FLASH_ADDR_INFO *fInfo);


#if defined(ATI_BSP_PERSONALITY)
#include <ati_hwdefs.h>
extern ATI_NVDATA atiNvramData;
extern int NvRamLookUpTable[1024][3];
int AtiTagData[4][3];
unsigned *atiPersistentLogAddr=NULL;
unsigned atiPersistentLogSize=0;
unsigned *atiKernelBootReasonAddr=NULL;
extern tAtiReservedMemoryMap *atiResMemMap;
extern tAtiPersistentLogMemoryMap *atiPersLogMemMap;
#endif

/* Typedefs. */

#if defined (WIRELESS)
#define SES_EVENT_BTN_PRESSED      0x00000001
#define SES_EVENTS                 SES_EVENT_BTN_PRESSED /*OR all values if any*/
#define SES_LED_OFF                0
#define SES_LED_ON                 1
#define SES_LED_BLINK              2

#if defined(CONFIG_BCM96362) || defined(CONFIG_BCM963268) || defined(CONFIG_BCM96318) || defined(CONFIG_BCM96828)
#define WLAN_ONBOARD_SLOT	WLAN_ONCHIP_DEV_SLOT
#else
#define WLAN_ONBOARD_SLOT       1 /* Corresponds to IDSEL -- EBI_A11/PCI_AD12 */
#endif

#define BRCM_VENDOR_ID       0x14e4
#define BRCM_WLAN_DEVICE_IDS 0x4300
#define BRCM_WLAN_DEVICE_IDS_DEC 43

#define WLAN_ON   1
#define WLAN_OFF  0
#endif

typedef struct
{
    unsigned long ulId;
    char chInUse;
    char chReserved[3];
} MAC_ADDR_INFO, *PMAC_ADDR_INFO;

typedef struct
{
    unsigned long ulNumMacAddrs;
    unsigned char ucaBaseMacAddr[NVRAM_MAC_ADDRESS_LEN];
    MAC_ADDR_INFO MacAddrs[1];
} MAC_INFO, *PMAC_INFO;

typedef struct
{
    unsigned char gponSerialNumber[NVRAM_GPON_SERIAL_NUMBER_LEN];
    unsigned char gponPassword[NVRAM_GPON_PASSWORD_LEN];
} GPON_INFO, *PGPON_INFO;

typedef struct
{
    unsigned long eventmask;
} BOARD_IOC, *PBOARD_IOC;


/*Dyinggasp callback*/
typedef void (*cb_dgasp_t)(void *arg);
typedef struct _CB_DGASP__LIST
{
    struct list_head list;
    char name[IFNAMSIZ];
    cb_dgasp_t cb_dgasp_fn;
    void *context;
}CB_DGASP_LIST , *PCB_DGASP_LIST;


/* Externs. */
extern struct file *fget_light(unsigned int fd, int *fput_needed);
extern unsigned long getMemorySize(void);
extern void __init boardLedInit(void);
extern void boardLedCtrl(BOARD_LED_NAME, BOARD_LED_STATE);

#if defined(ATI_BSP_PERSONALITY)
PFILE_TAG kerSysImageTagCpy(int imageNumber, PFILE_TAG pTag);
#endif

/* Prototypes. */
static void set_mac_info( void );
static void set_gpon_info( void );
static int board_open( struct inode *inode, struct file *filp );
static int board_ioctl( struct inode *inode, struct file *flip, unsigned int command, unsigned long arg );
static ssize_t board_read(struct file *filp,  char __user *buffer, size_t count, loff_t *ppos);
static unsigned int board_poll(struct file *filp, struct poll_table_struct *wait);
static int board_release(struct inode *inode, struct file *filp);

static BOARD_IOC* borad_ioc_alloc(void);
static void borad_ioc_free(BOARD_IOC* board_ioc);

/*
 * flashImageMutex must be acquired for all write operations to
 * nvram, CFE, or fs+kernel image.  (cfe and nvram may share a sector).
 */
DEFINE_MUTEX(flashImageMutex);
static void writeNvramDataCrcLocked(PNVRAM_DATA pNvramData);
static PNVRAM_DATA readNvramData(void);

/* DyingGasp function prototype */
static irqreturn_t kerSysDyingGaspIsr(int irq, void * dev_id);
static void __init kerSysInitDyingGaspHandler( void );
static void __exit kerSysDeinitDyingGaspHandler( void );
/* -DyingGasp function prototype - */
/* dgaspMutex protects list add and delete, but is ignored during isr. */
static DEFINE_MUTEX(dgaspMutex);

static int ConfigCs(BOARD_IOCTL_PARMS *parms);


#if defined (WIRELESS)
static irqreturn_t sesBtn_isr(int irq, void *dev_id);
static void __init sesBtn_mapIntr(int context);
static Bool sesBtn_pressed(void);
static unsigned int sesBtn_poll(struct file *file, struct poll_table_struct *wait);
static ssize_t sesBtn_read(struct file *file,  char __user *buffer, size_t count, loff_t *ppos);
static void __init sesLed_mapGpio(void);
static void sesLed_ctrl(int action);
static void __init ses_board_init(void);
static void __exit ses_board_deinit(void);
static void __init kerSysScreenPciDevices(void);
static void kerSetWirelessPD(int state);
#endif

#if defined(PCIE_BASE)
static void __init kerSysCheckPowerDownPcie(void);
#endif

static void str_to_num(char* in, char *out, int len);
static int add_proc_files(void);
static int del_proc_files(void);
static int proc_get_param(char *page, char **start, off_t off, int cnt, int *eof, void *data);
static int proc_set_param(struct file *f, const char *buf, unsigned long cnt, void *data);
static int proc_set_led(struct file *f, const char *buf, unsigned long cnt, void *data);
static int proc_set_led(struct file *f, const char *buf, unsigned long cnt, void *data);
#if 1 //ATI_BSP_PERSONALITY
static int proc_get_led(char *page, char **start, off_t off, int cnt, int *eof, void *data);
static int proc_get_gpio(char *page, char **start, off_t off, int cnt, int *eof, void *data);
static int proc_set_gpio(struct file *f, const char *buf, unsigned long cnt, void *data);
static int proc_get_gpio_dir(char *page, char **start, off_t off, int cnt, int *eof, void *data);
static int proc_set_gpio_dir(struct file *f, const char *buf, unsigned long cnt, void *data);
#endif

static irqreturn_t reset_isr(int irq, void *dev_id);

// macAddrMutex is used by kerSysGetMacAddress and kerSysReleaseMacAddress
// to protect access to g_pMacInfo
static DEFINE_MUTEX(macAddrMutex);
static PMAC_INFO g_pMacInfo = NULL;
static PGPON_INFO g_pGponInfo = NULL;
static unsigned long g_ulSdramSize;
#if defined(CONFIG_BCM96368)
static unsigned long g_ulSdramWidth;
#endif
static int g_ledInitialized = 0;
static wait_queue_head_t g_board_wait_queue;
static CB_DGASP_LIST *g_cb_dgasp_list_head = NULL;

#define MAX_PAYLOAD_LEN 64
static struct sock *g_monitor_nl_sk;
static int g_monitor_nl_pid = 0 ;
static void kerSysInitMonitorSocket( void );
static void kerSysCleanupMonitorSocket( void );

#if defined(CONFIG_BCM96368)
static void ChipSoftReset(void);
static void ResetPiRegisters( void );
static void PI_upper_set( volatile uint32 *PI_reg, int newPhaseInt );
static void PI_lower_set( volatile uint32 *PI_reg, int newPhaseInt );
static void TurnOffSyncMode( void );
#endif

#if defined(CONFIG_BCM96816)
void board_Init6829( void );
#endif

#if defined(ATI_PRODUCT_CONFIG) && (defined(CONFIG_ATI_SFP) || defined(CONFIG_BCM_I2C_BUS))
#define SFP_POLLING_INTERVAL 2*HZ

extern int atlat_sfp_init(void);
extern int atlat_sfp_add_sfp(struct i2c_adapter *adap, int boardIndex, int portIndex);
extern int atlat_sfp_delete_sfp(int boardIndex, int portIndex);

static int get_i2c_sfp_gpio_numbers(unsigned int *scl_pin, 
                                    unsigned int *scl_pin_found,
                                    unsigned int *sda_pin,
                                    unsigned int *sda_pin_found,
                                    unsigned int *pres_pin,
                                    unsigned int *pres_pin_found);

static void sfp_work_queue_handler(struct work_struct *work);

static int get_sfp_vport_number_from_AtiEthSwitchMap(void);

#if !defined(CONFIG_BCM_I2C_BUS)
/* 
 * Configure i2c-ati-gpio driver. 
 * SDA and SCL GPIO numbers will be configured at run time 
 * based on GPIO variables defined in UBOOT. 
 */
static struct i2c_ati_gpio_platform_data i2c_ati_gpio_pdata = {
    .sda_pin =              0,
    .sda_is_open_drain =    0,
    .scl_pin =              0,
    .scl_is_open_drain =    0,
    .scl_is_output_only =   1,
    .udelay =               10,
};

static struct platform_device i2c_ati_gpio_device = {
    .name = "i2c-ati-gpio",
    .id   = -1,
    .dev.platform_data = &i2c_ati_gpio_pdata,
};
#endif

typedef struct {
    struct i2c_adapter  *i2c_adap;
    unsigned int        sfp_pres_gpio;
    unsigned int        sfp_board_index;
    unsigned int        sfp_port_index;
    struct delayed_work sfp_work;
    unsigned int        poll_interval;  /* SFP presense polling interval in jiffies */
} sfp_info_t;

sfp_info_t sfp_info = {0};
#endif

static kerSysMacAddressNotifyHook_t kerSysMacAddressNotifyHook = NULL;

/* restore default work structure */
static struct work_struct restoreDefaultWork;

static struct file_operations board_fops =
{
    open:       board_open,
    ioctl:      board_ioctl,
    poll:       board_poll,
    read:       board_read,
    release:    board_release,
};

uint32 board_major = 0;

#if defined (WIRELESS)
static unsigned short sesBtn_irq = BP_NOT_DEFINED;
static unsigned short sesLed_gpio = BP_NOT_DEFINED;
#endif

#if defined(MODULE)
int init_module(void)
{
    return( brcm_board_init() );
}

void cleanup_module(void)
{
    if (MOD_IN_USE)
        printk("brcm flash: cleanup_module failed because module is in use\n");
    else
        brcm_board_cleanup();
}
#endif //MODULE

static int map_external_irq (int irq)
{
    int map_irq;

    switch (irq) {
    case BP_EXT_INTR_0   :
        map_irq = INTERRUPT_ID_EXTERNAL_0;
        break ;
    case BP_EXT_INTR_1   :
        map_irq = INTERRUPT_ID_EXTERNAL_1;
        break ;
    case BP_EXT_INTR_2   :
        map_irq = INTERRUPT_ID_EXTERNAL_2;
        break ;
    case BP_EXT_INTR_3   :
        map_irq = INTERRUPT_ID_EXTERNAL_3;
        break ;
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816) || defined(CONFIG_BCM96818)
    case BP_EXT_INTR_4   :
        map_irq = INTERRUPT_ID_EXTERNAL_4;
        break ;
    case BP_EXT_INTR_5   :
        map_irq = INTERRUPT_ID_EXTERNAL_5;
        break ;
#endif
    default           :
        printk ("Invalid External Interrupt definition \n") ;
        map_irq = 0 ;
        break ;
    }
    return (map_irq) ;
}

/* A global variable used by Power Management and other features to determine if Voice is idle or not */
volatile int isVoiceIdle = 1;
EXPORT_SYMBOL(isVoiceIdle);

#if defined(CONFIG_BCM_AVS_PWRSAVE)
/* A generic kernel thread for board related maintenance activities */
/* Only used if AVS is configured, but feel free to change */
static DECLARE_COMPLETION(poll_done);
static atomic_t poll_lock = ATOMIC_INIT(1);
static int poll_pid = -1;

/* The AVS algorith lowers 1V2 until the slowest ring osc (the one with
   the largest count) reaches the RING_OSC_LOWER_BOUND. Then the algorithm
   will re-increase the voltage if the ring osc continues to slow down to a
   point where its count exceeds the RING_OSC_UPPER_BOUND. */

#if defined(CONFIG_BCM_PWRMNGT_MODULE)
int AvsEnabled = -1; // Wait for the module to control if it is enabled or not
#else
int AvsEnabled = 1; // There is no control so force it to be enabled
#endif
void kerSysBcmEnableAvs(int enable)
{
    unsigned short vregVsel1P2;

    /* Set target 1V2 level */
    if (BpGetVregSel1P2(&vregVsel1P2) == BP_SUCCESS ) {
        AvsEnabled = -1;
        printk("Adaptive Voltage Scaling is disabled because 1V2 is forced to a specific level by design\n");
    } else {
        AvsEnabled = enable;
        printk("Adaptive Voltage Scaling is now %s\n", (enable==1?"enabled":(enable==0?"disabled":"stopped")));
    }
}

int kerSysBcmAvsEnabled(void)
{
    return AvsEnabled;
}

#if defined(CONFIG_BCM96362)
/* 36483, Highest ring osc count read during successful PVT
   a small margin is included to avoid ever reaching this highest count */
//#define RING_OSC_UPPER_BOUND (0x8E83 - 100)
#define RING_OSC_UPPER_BOUND (0x6F1C - 100) // 3.6 MHz + tidbits
/* 1100 is about the max ring osc variation when doing a 1V2 step change */
#define RING_OSC_LOWER_BOUND (RING_OSC_UPPER_BOUND - 1100)
/* 1V2 is set by default to 1.225. HW team is asking that we don't allow the
   voltage to go below 1.19 V. */
#define VREG_VSEL1P2_LOWER_BOUND ((VREG_VSEL1P2_MIDDLE+1) - 3)
/* 6362 Slow parts need to have their voltage increased to avoid WLAN issues
   This is the threshold we use to identify what looks like a slow part.
   A large enough delta is needed between this threshold and the 
   RING_OSC_UPPER_BOUND (~2 MHz here) */
#define RING_OSC_SS_UPPER_BOUND (0x75A5) // 3.4 MHz
#define VREG_VSEL1P2_SS_TARGET   (VREG_VSEL1P2_MIDDLE+1)
#elif defined(CONFIG_BCM96816)
/* PVT suggests we can go as high as A6AA but choosing to be conservative */
#define RING_OSC_UPPER_BOUND (0x9000 - 100)
#define RING_OSC_LOWER_BOUND (RING_OSC_UPPER_BOUND - 1100)
#define VREG_VSEL1P2_LOWER_BOUND ((VREG_VSEL1P2_MIDDLE+1) - 4)
#elif defined(CONFIG_BCM96368)
#define RING_OSC_UPPER_BOUND (0x84FA - 100)
#define RING_OSC_LOWER_BOUND (RING_OSC_UPPER_BOUND - 1500)
#define VREG_VSEL1P2_LOWER_BOUND  ((VREG_VSEL1P2_MIDDLE+1) - 4)
#elif defined(CONFIG_BCM96818)
#define RING_OSC_UPPER_BOUND (0x9800 - 100)
#define RING_OSC_LOWER_BOUND (RING_OSC_UPPER_BOUND - 1100)
#define VREG_VSEL1P2_LOWER_BOUND (MISC_VREG_CONTROL1_VSEL1P2_DEFAULT - 0x18)
#else
#error "Not Implemented for this chip"
#endif

#define AVSDEBUG(f, ...)
//#define AVSDEBUG(f, ...) printk(f, __VA_ARGS__)

static void brcm_adaptive_voltage_scaling(void)
{
    static int max_count = 0;
    static int is_ss_part = 0;
    int ring_osc_select = 0;
    int current_1v2 = 0;
    int next_1v2 = 0;
    uint32 RingOscCtrl1 = GPIO->RingOscCtrl1;

    /* Verify is AVS is not forced off */
    if (AvsEnabled != -1) {
#if !(defined(CONFIG_BCM96368) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM96328) || defined(CONFIG_BCM96816))
        static uint32 initmiscVregCtrl1 = 0;

        if (!initmiscVregCtrl1) {
            if (!(MISC->miscVregCtrl0 & MISC_VREG_CONTROL0_REG_RESET_B)) {
                MISC->miscVregCtrl2 &= ~MISC_VREG_CONTROL2_SWITCH_CLKEN;
                MISC->miscVregCtrl1 = MISC->miscVregCtrl1;
                MISC->miscVregCtrl0 |= MISC_VREG_CONTROL0_REG_RESET_B;
                initmiscVregCtrl1 = 1;
            }
        }
#endif
        /* Verify if the ring oscillator has completed a measurement */
        /* This will only fail on the very first call to this function */
        if (RingOscCtrl1 & RING_OSC_IRQ)
        {
            AVSDEBUG("Read ring osc %ld: %lx\n",
                    (RingOscCtrl1 & RING_OSC_SELECT_MASK) >> RING_OSC_SELECT_SHIFT,
                     RingOscCtrl1 & RING_OSC_COUNT_MASK);
            if ((RingOscCtrl1 & RING_OSC_COUNT_MASK) > max_count)
            {
                max_count = RingOscCtrl1 & RING_OSC_COUNT_MASK;
                AVSDEBUG("max_count: %x\n", max_count);
            }

            /* Move to the next enabled ring osc */
            ring_osc_select = (RingOscCtrl1 & RING_OSC_SELECT_MASK) >> RING_OSC_SELECT_SHIFT;
            while (++ring_osc_select < RING_OSC_MAX)
            {
                if ((((1<<ring_osc_select)<<RING_OSC_ENABLE_SHIFT) & RING_OSC_ENABLE_MASK) != 0)
                {
                    break;
                }
            }

            /* If we have read all ring osc, determine if the voltage should be changed */
            if (ring_osc_select == RING_OSC_MAX)
            {
                /* All ring osc have been read, prepare for the next round */
                /* 0 is always a valid ring osc so no need to verify if it is enabled */
                ring_osc_select = 0;

                /* Check if the voltage should be adjusted */
                if ((max_count < RING_OSC_LOWER_BOUND) && AvsEnabled && !is_ss_part)
                {
                    /* The ring osc is too fast, reduce the voltage if it is not too low */
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816)
                    current_1v2 = (GPIO->VregConfig & VREG_VSEL1P2_MASK) >> VREG_VSEL1P2_SHIFT;
#elif defined(CONFIG_BCM96328)
                    current_1v2 = (MISC->miscVregCtrl0 & VREG_VSEL1P2_MASK) >> VREG_VSEL1P2_SHIFT;
#else
                    current_1v2 = (MISC->miscVregCtrl1 & VREG_VSEL1P2_MASK) >> VREG_VSEL1P2_SHIFT;
#endif
                    next_1v2 = current_1v2;
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96362) || defined(CONFIG_BCM96328) || defined(CONFIG_BCM96816)
                    if (current_1v2 == 0)
                    {
                        next_1v2 = VREG_VSEL1P2_MIDDLE;
                    }
                    else if (current_1v2 > VREG_VSEL1P2_LOWER_BOUND)
                    {
                        next_1v2--;
                    }
#else
                    if (current_1v2 > VREG_VSEL1P2_LOWER_BOUND)
                    {
                        next_1v2 -= 8;
                    }
#endif
                    AVSDEBUG("ring_osc is fast, can reduce voltage: %d to %d\n", current_1v2, next_1v2);
                }
                else if ((max_count > RING_OSC_UPPER_BOUND) || !AvsEnabled || is_ss_part)
                {
                    /* The ring osc is too slow, increase the voltage up to the default of 0 */
                    /* If AVS is disabled, we need to force the voltage to come back up to default */
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816)
                    current_1v2 = (GPIO->VregConfig & VREG_VSEL1P2_MASK) >> VREG_VSEL1P2_SHIFT;
#elif defined(CONFIG_BCM96328)
                    current_1v2 = (MISC->miscVregCtrl0 & VREG_VSEL1P2_MASK) >> VREG_VSEL1P2_SHIFT;
#else
                    current_1v2 = (MISC->miscVregCtrl1 & VREG_VSEL1P2_MASK) >> VREG_VSEL1P2_SHIFT;
#endif
                    next_1v2 = current_1v2;

#if defined(CONFIG_BCM96362)
                    /* On 6362, we try to identify SS parts to increase their voltage
                       to help WLAN performance */
                    if (AvsEnabled && (max_count > RING_OSC_SS_UPPER_BOUND))
                    {
                        is_ss_part = 1;
                    }

                    if (is_ss_part)
                    {
                        if (current_1v2 == VREG_VSEL1P2_MIDDLE)
                        {
                            next_1v2 = 0;
                        }
                        else if (current_1v2 == 0)
                        {
                            next_1v2 = VREG_VSEL1P2_MIDDLE+1;
                        }
                        else if (current_1v2 < VREG_VSEL1P2_SS_TARGET)
                        {
                            next_1v2++;
                        }
                    }
                    else
                    {
                        if (current_1v2 == VREG_VSEL1P2_MIDDLE)
                        {
                            next_1v2 = 0;
                        }
                        else if (current_1v2 != 0)
                        {
                            next_1v2++;
                        }
                    }
#elif defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816) || defined(CONFIG_BCM96328)
                    if (current_1v2 == VREG_VSEL1P2_MIDDLE)
                    {
                        next_1v2 = 0;
                    }
                    else if (current_1v2 != 0)
                    {
                        next_1v2++;
                    }
#else
                    if (current_1v2 < MISC_VREG_CONTROL1_VSEL1P2_DEFAULT)
                    {
                        next_1v2 += 8;
                    }
#endif
                    AVSDEBUG("ring_osc is slow, can increase voltage: %d to %d\n", current_1v2, next_1v2);
                }

                if (next_1v2 != current_1v2)
                {
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816)
                    GPIO->VregConfig = (GPIO->VregConfig & ~VREG_VSEL1P2_MASK) | (next_1v2 << VREG_VSEL1P2_SHIFT);
#elif defined(CONFIG_BCM96328) 
                    MISC->miscVregCtrl0 = (MISC->miscVregCtrl0 & ~VREG_VSEL1P2_MASK) | (next_1v2 << VREG_VSEL1P2_SHIFT);
#else
                    MISC->miscVregCtrl1 = (MISC->miscVregCtrl1 & ~VREG_VSEL1P2_MASK) | (next_1v2 << VREG_VSEL1P2_SHIFT);
#endif
                    AVSDEBUG("Adjusted voltage: %d to %d\n", current_1v2, next_1v2);
                }
                max_count = 0;
            }
        }         

        /* Start a new ring osc count cycle by resetting the counter */
        GPIO->RingOscCtrl1 = RING_OSC_ENABLE_MASK |
                             RING_OSC_COUNT_RESET;
        GPIO->RingOscCtrl1 = RING_OSC_ENABLE_MASK |
                             (ring_osc_select << RING_OSC_SELECT_SHIFT);
        /* Writing to this register starts the count */
        GPIO->RingOscCtrl0 = RING_OSC_512_CYCLES;
    }
}

static int brcm_board_timer( void *data )
{
    daemonize("board-timer");

    while (atomic_read(&poll_lock) > 0)
    {
        brcm_adaptive_voltage_scaling();

        /* Sleep for 1 second (HZ jiffies) */
        set_current_state(TASK_INTERRUPTIBLE);
        schedule_timeout(HZ);
    }

    complete_and_exit(&poll_done, 0);
    printk("brcm_board_timer: thread exits!\n");

}
#endif

#if defined(CONFIG_BCM96368)
static unsigned long getMemoryWidth(void)
{
    unsigned long memCfg;

    memCfg = MEMC->Config;
    memCfg &= MEMC_WIDTH_MASK;
    memCfg >>= MEMC_WIDTH_SHFT;

    return memCfg;
}
#endif

static int __init brcm_board_init( void )
{
    unsigned short rstToDflt_irq;
    int ret;
    bcmLogSpiCallbacks_t loggingCallbacks;

    ret = register_chrdev(BOARD_DRV_MAJOR, "brcmboard", &board_fops );
    if (ret < 0)
        printk( "brcm_board_init(major %d): fail to register device.\n",BOARD_DRV_MAJOR);
    else
    {
        printk("brcmboard: brcm_board_init entry\n");
        board_major = BOARD_DRV_MAJOR;

        g_ulSdramSize = getMemorySize();
#if defined(CONFIG_BCM96368)
        g_ulSdramWidth = getMemoryWidth();
#endif
        set_mac_info();
        set_gpon_info();

        init_waitqueue_head(&g_board_wait_queue);
#if defined (WIRELESS)
        kerSysScreenPciDevices();
        ses_board_init();
        kerSetWirelessPD(WLAN_ON);
#endif
#if defined(PCIE_BASE)
        kerSysCheckPowerDownPcie();
#endif
        kerSysInitMonitorSocket();
        kerSysInitDyingGaspHandler();

        boardLedInit();
        g_ledInitialized = 1;

        if( BpGetResetToDefaultExtIntr(&rstToDflt_irq) == BP_SUCCESS )
        {
            rstToDflt_irq = map_external_irq (rstToDflt_irq) ;
            BcmHalMapInterrupt((FN_HANDLER)reset_isr, 0, rstToDflt_irq);
            BcmHalInterruptEnable(rstToDflt_irq);
        }

#if defined(CONFIG_BCM_CPLD1)
        // Reserve SPI bus to control external CPLD for Standby Timer
        BcmCpld1Initialize();
#endif

#if defined(CONFIG_BCM_AVS_PWRSAVE)
        poll_pid = kernel_thread(brcm_board_timer, NULL, CLONE_KERNEL);
#if !defined(CONFIG_BCM_PWRMNGT_MODULE)
    // Show that AVS is enabled when PWRMGNT control is not compiled in
    printk("Adaptive Voltage Scaling is always enabled\n");
#endif
#endif

    }

    add_proc_files();

#if defined(CONFIG_BCM96816)
    board_Init6829();
    loggingCallbacks.kerSysSlaveRead   = kerSysBcmSpiSlaveRead;
    loggingCallbacks.kerSysSlaveWrite  = kerSysBcmSpiSlaveWrite;
    loggingCallbacks.bpGet6829PortInfo = BpGet6829PortInfo;
#else
    loggingCallbacks.kerSysSlaveRead   = NULL;
    loggingCallbacks.kerSysSlaveWrite  = NULL;
    loggingCallbacks.bpGet6829PortInfo = NULL;
#endif
    loggingCallbacks.reserveSlave      = BcmSpiReserveSlave;
    loggingCallbacks.syncTrans         = BcmSpiSyncTrans;
    bcmLog_registerSpiCallbacks(loggingCallbacks);

#if defined(ATI_PRODUCT_CONFIG) && (defined(CONFIG_ATI_SFP) || defined(CONFIG_BCM_I2C_BUS))
    {
        unsigned int sclFound = 0, sdaFound = 0, presFound = 0;

        /*
         * Init SFP board and port indecies 
         */
        sfp_info.sfp_board_index = 1;
        sfp_info.sfp_port_index = get_sfp_vport_number_from_AtiEthSwitchMap();
    
#if !defined(CONFIG_BCM_I2C_BUS)
        /*
         * Figure out which GPIO to use for i2c-ati-gpio driver.
         */
        ret = get_i2c_sfp_gpio_numbers(&i2c_ati_gpio_pdata.scl_pin, &sclFound,
                                       &i2c_ati_gpio_pdata.sda_pin, &sdaFound,
                                       &(sfp_info.sfp_pres_gpio),   &presFound);
        if (ret != 0)
        {
            printk(KERN_ERR "AtiGpioSfpI2cClk, AtiGpioSfpI2cDat or AtiGpioSfpPresent definitions are not valid!\n");
            printk(KERN_ERR "Aborting SFP I2C driver init...\n");
            goto abort_sfp_init;
        }

        if (!sclFound || !sdaFound || !presFound)
        {
            printk(KERN_ERR "AtiGpioSfpI2cClk, AtiGpioSfpI2cDat or AtiGpioSfpPresent definition not found!\n");
            printk(KERN_ERR "Aborting SFP I2C driver init...\n");
            goto abort_sfp_init;
        }

        ret = platform_device_register(&i2c_ati_gpio_device);
        if (ret != 0)
        {
            printk(KERN_ERR "Could not register i2c-ati-gpio driver!\n");
            goto abort_sfp_init;
        }
#else
        {
            unsigned int scl_pin = 0, sda_pin = 0;

            ret = get_i2c_sfp_gpio_numbers(&scl_pin, &sclFound,
                                           &sda_pin, &sdaFound,
                                           &sfp_info.sfp_pres_gpio, &presFound);
            if (ret != 0)
            {
                printk(KERN_ERR "AtiGpioSfpPresent definition is not valid!\n");
                printk(KERN_ERR "Aborting SFP I2C driver init...\n");
                goto abort_sfp_init;
            }
        }
        if (!presFound)
        {
            printk(KERN_ERR "AtiGpioSfpPresent definition not found!\n");
            printk(KERN_ERR "Aborting SFP I2C driver init...\n");
            goto abort_sfp_init;
        }
#endif
    
        ret = atlat_sfp_init();
        if (ret != 0)
        {
            printk(KERN_ERR "Could not init SFP bus!\n");
            goto abort_sfp_init;
        }
    }
abort_sfp_init:
#endif
    return ret;
}

#if defined(ATI_PRODUCT_CONFIG) && (defined(CONFIG_ATI_SFP) || defined(CONFIG_BCM_I2C_BUS))
/*
 * The board module init has been split into two pieces: brcm_board_init() and this 
 * function  brcm_board_sfp_init(). brcm_board_init() is loaded early. In fact, it is loaded 
 * before the the I2C adapter driver has had a chance to initialize and register with i2c core. 
 * So, i2c adapter query has moved to this function, which is called later in the driver 
 * init chain due to late_initcall(). 
 */
static int __init brcm_board_sfp_init( void )
{
    sfp_info.i2c_adap = i2c_get_adapter(0);
    if (!sfp_info.i2c_adap)
    {
        printk(KERN_ERR "Could not get I2C adapter!\n");
        return -1;
    }
    /*
     * Setup a workqueue item to poll the SFP presense GPIO. 
     */
    sfp_info.poll_interval = SFP_POLLING_INTERVAL;
    INIT_DELAYED_WORK(&sfp_info.sfp_work, sfp_work_queue_handler);
    schedule_delayed_work(&sfp_info.sfp_work, sfp_info.poll_interval);
    return 0;
}
late_initcall(brcm_board_sfp_init);
#endif

static void __init set_mac_info( void )
{
    NVRAM_DATA *pNvramData;
    unsigned long ulNumMacAddrs;

    if (NULL == (pNvramData = readNvramData()))
    {
        printk("set_mac_info: could not read nvram data\n");
        return;
    }

    ulNumMacAddrs = pNvramData->ulNumMacAddrs;

    if( ulNumMacAddrs > 0 && ulNumMacAddrs <= NVRAM_MAC_COUNT_MAX )
    {
        unsigned long ulMacInfoSize =
            sizeof(MAC_INFO) + ((sizeof(MAC_ADDR_INFO)) * (ulNumMacAddrs-1));

        g_pMacInfo = (PMAC_INFO) kmalloc( ulMacInfoSize, GFP_KERNEL );

        if( g_pMacInfo )
        {
            memset( g_pMacInfo, 0x00, ulMacInfoSize );
            g_pMacInfo->ulNumMacAddrs = pNvramData->ulNumMacAddrs;
            memcpy( g_pMacInfo->ucaBaseMacAddr, pNvramData->ucaBaseMacAddr,
                NVRAM_MAC_ADDRESS_LEN );
        }
        else
            printk("ERROR - Could not allocate memory for MAC data\n");
    }
    else
        printk("ERROR - Invalid number of MAC addresses (%ld) is configured.\n",
        ulNumMacAddrs);
    kfree(pNvramData);
}

static int gponParamsAreErased(NVRAM_DATA *pNvramData)
{
    int i;
    int erased = 1;

    for(i=0; i<NVRAM_GPON_SERIAL_NUMBER_LEN-1; ++i) {
        if((pNvramData->gponSerialNumber[i] != (char) 0xFF) &&
            (pNvramData->gponSerialNumber[i] != (char) 0x00)) {
                erased = 0;
                break;
        }
    }

    if(!erased) {
        for(i=0; i<NVRAM_GPON_PASSWORD_LEN-1; ++i) {
            if((pNvramData->gponPassword[i] != (char) 0xFF) &&
                (pNvramData->gponPassword[i] != (char) 0x00)) {
                    erased = 0;
                    break;
            }
        }
    }

    return erased;
}

static void __init set_gpon_info( void )
{
    NVRAM_DATA *pNvramData;

    if (NULL == (pNvramData = readNvramData()))
    {
        printk("set_gpon_info: could not read nvram data\n");
        return;
    }

    g_pGponInfo = (PGPON_INFO) kmalloc( sizeof(GPON_INFO), GFP_KERNEL );

    if( g_pGponInfo )
    {
        if ((pNvramData->ulVersion < NVRAM_FULL_LEN_VERSION_NUMBER) ||
            gponParamsAreErased(pNvramData))
        {
            strcpy( g_pGponInfo->gponSerialNumber, DEFAULT_GPON_SN );
            strcpy( g_pGponInfo->gponPassword, DEFAULT_GPON_PW );
        }
        else
        {
            strncpy( g_pGponInfo->gponSerialNumber, pNvramData->gponSerialNumber,
                NVRAM_GPON_SERIAL_NUMBER_LEN );
            g_pGponInfo->gponSerialNumber[NVRAM_GPON_SERIAL_NUMBER_LEN-1]='\0';
            strncpy( g_pGponInfo->gponPassword, pNvramData->gponPassword,
                NVRAM_GPON_PASSWORD_LEN );
            g_pGponInfo->gponPassword[NVRAM_GPON_PASSWORD_LEN-1]='\0';
        }
    }
    else
    {
        printk("ERROR - Could not allocate memory for GPON data\n");
    }
    kfree(pNvramData);
}

void __exit brcm_board_cleanup( void )
{
    printk("brcm_board_cleanup()\n");
    del_proc_files();

    if (board_major != -1)
    {
#if defined (WIRELESS)
        ses_board_deinit();
#endif
        kerSysDeinitDyingGaspHandler();
        kerSysCleanupMonitorSocket();
        unregister_chrdev(board_major, "board_ioctl");

#if defined(CONFIG_BCM_AVS_PWRSAVE)
        if (poll_pid >= 0)
        {
            atomic_dec(&poll_lock);
            wait_for_completion(&poll_done);
        }
#endif
    }
#if defined(ATI_PRODUCT_CONFIG) && (defined(CONFIG_ATI_SFP) || defined(CONFIG_BCM_I2C_BUS))
        cancel_delayed_work(&sfp_info.sfp_work);
#endif
}

static BOARD_IOC* borad_ioc_alloc(void)
{
    BOARD_IOC *board_ioc =NULL;
    board_ioc = (BOARD_IOC*) kmalloc( sizeof(BOARD_IOC) , GFP_KERNEL );
    if(board_ioc)
    {
        memset(board_ioc, 0, sizeof(BOARD_IOC));
    }
    return board_ioc;
}

static void borad_ioc_free(BOARD_IOC* board_ioc)
{
    if(board_ioc)
    {
        kfree(board_ioc);
    }
}


static int board_open( struct inode *inode, struct file *filp )
{
    filp->private_data = borad_ioc_alloc();

    if (filp->private_data == NULL)
        return -ENOMEM;

    return( 0 );
}

static int board_release(struct inode *inode, struct file *filp)
{
    BOARD_IOC *board_ioc = filp->private_data;

    wait_event_interruptible(g_board_wait_queue, 1);
    borad_ioc_free(board_ioc);

    return( 0 );
}


static unsigned int board_poll(struct file *filp, struct poll_table_struct *wait)
{
    unsigned int mask = 0;
#if defined (WIRELESS)
    BOARD_IOC *board_ioc = filp->private_data;
#endif

    poll_wait(filp, &g_board_wait_queue, wait);
#if defined (WIRELESS)
    if(board_ioc->eventmask & SES_EVENTS){
        mask |= sesBtn_poll(filp, wait);
    }
#endif

    return mask;
}

static ssize_t board_read(struct file *filp,  char __user *buffer, size_t count, loff_t *ppos)
{
#if defined (WIRELESS)
    BOARD_IOC *board_ioc = filp->private_data;
    if(board_ioc->eventmask & SES_EVENTS){
        return sesBtn_read(filp, buffer, count, ppos);
    }
#endif
    return 0;
}

/***************************************************************************
// Function Name: getCrc32
// Description  : caculate the CRC 32 of the given data.
// Parameters   : pdata - array of data.
//                size - number of input data bytes.
//                crc - either CRC32_INIT_VALUE or previous return value.
// Returns      : crc.
****************************************************************************/
#if defined(ATI_BSP_PERSONALITY)
#define DO1(buf) crc = Crc32_table[((int)crc ^ (*buf++)) & 0xff] ^ (crc >> 8);
#define DO2(buf)  DO1(buf); DO1(buf);
#define DO4(buf)  DO2(buf); DO2(buf);
#define DO8(buf)  DO4(buf); DO4(buf);
#endif
static UINT32 getCrc32(byte *pdata, UINT32 size, UINT32 crc)
{
#if defined(ATI_BSP_PERSONALITY)
    crc = crc ^ 0xffffffffL;
    while (size >= 8)
    {
      DO8(pdata);
      size -= 8;
    }
    if (size) do {
      DO1(pdata);
    } while (--size);
    return crc ^ 0xffffffffL;
#else
    while (size-- > 0)
        crc = (crc >> 8) ^ Crc32_table[(crc ^ *pdata++) & 0xff];

    return crc;
#endif
}

/** calculate the CRC for the nvram data block and write it to flash.
 * Must be called with flashImageMutex held.
 */
static void writeNvramDataCrcLocked(PNVRAM_DATA pNvramData)
{
    UINT32 crc = CRC32_INIT_VALUE;

#if defined(ATI_BSP_PERSONALITY)
     // NVRAM is not in CFE application sector - do not write
    return;
#endif
    BCM_ASSERT_HAS_MUTEX_C(&flashImageMutex);

    pNvramData->ulCheckSum = 0;
    crc = getCrc32((char *)pNvramData, sizeof(NVRAM_DATA), crc);
    pNvramData->ulCheckSum = crc;
    kerSysNvRamSet((char *)pNvramData, sizeof(NVRAM_DATA), 0);
}


/** read the nvramData struct from the in-memory copy of nvram.
 * The caller is not required to have flashImageMutex when calling this
 * function.  However, if the caller is doing a read-modify-write of
 * the nvram data, then the caller must hold flashImageMutex.  This function
 * does not know what the caller is going to do with this data, so it
 * cannot assert flashImageMutex held or not when this function is called.
 *
 * @return pointer to NVRAM_DATA buffer which the caller must free
 *         or NULL if there was an error
 */
static PNVRAM_DATA readNvramData(void)
{
    UINT32 crc = CRC32_INIT_VALUE, savedCrc;
    NVRAM_DATA *pNvramData;

    // use GFP_ATOMIC here because caller might have flashImageMutex held
    if (NULL == (pNvramData = kmalloc(sizeof(NVRAM_DATA), GFP_ATOMIC)))
    {
        printk("readNvramData: could not allocate memory\n");
        return NULL;
    }

    kerSysNvRamGet((char *)pNvramData, sizeof(NVRAM_DATA), 0);
    savedCrc = pNvramData->ulCheckSum;
    pNvramData->ulCheckSum = 0;
    crc = getCrc32((char *)pNvramData, sizeof(NVRAM_DATA), crc);
    printk("inMemNvramData.ulCheckSum=%lx, %lx\n", savedCrc, crc);
#if 0
    if (savedCrc != crc)
    {
        // this can happen if we write a new cfe image into flash.
        // The new image will have an invalid nvram section which will
        // get updated to the inMemNvramData.  We detect it here and
        // commonImageWrite will restore previous copy of nvram data.
        kfree(pNvramData);
        pNvramData = NULL;
    }
#endif
    return pNvramData;
}



//**************************************************************************************
// Utitlities for dump memory, free kernel pages, mips soft reset, etc.
//**************************************************************************************

/***********************************************************************
* Function Name: dumpaddr
* Description  : Display a hex dump of the specified address.
***********************************************************************/
void dumpaddr( unsigned char *pAddr, int nLen )
{
    static char szHexChars[] = "0123456789abcdef";
    char szLine[80];
    char *p = szLine;
    unsigned char ch, *q;
    int i, j;
    unsigned long ul;

    while( nLen > 0 )
    {
        sprintf( szLine, "%8.8lx: ", (unsigned long) pAddr );
        p = szLine + strlen(szLine);

        for(i = 0; i < 16 && nLen > 0; i += sizeof(long), nLen -= sizeof(long))
        {
            ul = *(unsigned long *) &pAddr[i];
            q = (unsigned char *) &ul;
            for( j = 0; j < sizeof(long); j++ )
            {
                *p++ = szHexChars[q[j] >> 4];
                *p++ = szHexChars[q[j] & 0x0f];
                *p++ = ' ';
            }
        }

        for( j = 0; j < 16 - i; j++ )
            *p++ = ' ', *p++ = ' ', *p++ = ' ';

        *p++ = ' ', *p++ = ' ', *p++ = ' ';

        for( j = 0; j < i; j++ )
        {
            ch = pAddr[j];
            *p++ = (ch > ' ' && ch < '~') ? ch : '.';
        }

        *p++ = '\0';
        printk( "%s\r\n", szLine );

        pAddr += i;
    }
    printk( "\r\n" );
} /* dumpaddr */


/** this function actually does two things, stop other cpu and reset mips.
 * Kept the current name for compatibility reasons.  Image upgrade code
 * needs to call the two steps separately.
 */
void kerSysMipsSoftReset(void)
{
	unsigned long cpu;
	cpu = smp_processor_id();
	printk(KERN_INFO "kerSysMipsSoftReset: called on cpu %lu\n", cpu);

	stopOtherCpu();
	local_irq_disable();  // ignore interrupts, just execute reset code now
	resetPwrmgmtDdrMips();
}

extern void stop_other_cpu(void);  // in arch/mips/kernel/smp.c

void stopOtherCpu(void)
{
#if defined(CONFIG_SMP)
    stop_other_cpu();
#elif defined(CONFIG_BCM_ENDPOINT_MODULE) && defined(CONFIG_BCM_BCMDSP_MODULE)
    unsigned long cpu = (read_c0_diag3() >> 31) ? 0 : 1;

	// Disable interrupts on the other core and allow it to complete processing 
	// and execute the "wait" instruction
    printk(KERN_INFO "stopOtherCpu: stopping cpu %lu\n", cpu);	
    PERF->IrqControl[cpu].IrqMask = 0;
    mdelay(5);
#endif
}

void resetPwrmgmtDdrMips(void)
{
#if !defined (CONFIG_BCM96816) && !defined (CONFIG_BCM96818)
    // Power Management on Ethernet Ports may have brought down EPHY PLL
    // and soft reset below will lock-up 6362 if the PLL is not up
    // therefore bring it up here to give it time to stabilize
    GPIO->RoboswEphyCtrl &= ~EPHY_PWR_DOWN_DLL;
#endif

    // let UART finish printing
    udelay(100);


#if defined(CONFIG_BCM_CPLD1)
    // Determine if this was a request to enter Standby mode
    // If yes, this call won't return and a hard reset will occur later
    BcmCpld1CheckShutdownMode();
#endif

#if defined (CONFIG_BCM96368)
    {
        volatile int delay;
        volatile int i;
        local_irq_disable();
        // after we reset DRAM controller we can't access DRAM, so
        // the first iteration put things in i-cache and the scond interation do the actual reset
        for (i=0; i<2; i++) {
            DDR->DDR1_2PhaseCntl0 &= i - 1;
            DDR->DDR3_4PhaseCntl0 &= i - 1;

            if( i == 1 )
                ChipSoftReset();

            delay = 1000;
            while (delay--);
            PERF->pll_control |= SOFT_RESET*i;
            for(;i;) {} // spin mips and wait soft reset to take effect
        }
    }
#endif
#if !defined(CONFIG_BCM96328) && !defined(CONFIG_BCM96318)
#if defined (CONFIG_BCM96816) || defined(CONFIG_BCM96818)
    /* Work around reset issues */
    HVG_MISC_REG_CHANNEL_A->mask |= HVG_SOFT_INIT_0;
    HVG_MISC_REG_CHANNEL_B->mask |= HVG_SOFT_INIT_0;

    {
        unsigned char portInfo6829;
        /* for BHRGR board we need to toggle GPIO30 to
           reset - on early BHR baords this is the GPHY2
           link100 so setting it does not matter */
        if ( (BP_SUCCESS == BpGet6829PortInfo(&portInfo6829)) &&
             (0 != portInfo6829))
        {
            GPIO->GPIODir |= 1<<30;
            GPIO->GPIOio  &= ~(1<<30);
        }
    }
#endif
    PERF->pll_control |= SOFT_RESET;    // soft reset mips
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816) || defined(CONFIG_BCM96818)
    PERF->pll_control = 0;
#endif
#else
    TIMER->SoftRst = 1;
#endif
    for(;;) {} // spin mips and wait soft reset to take effect
}

unsigned long kerSysGetMacAddressType( unsigned char *ifName )
{
    unsigned long macAddressType = MAC_ADDRESS_ANY;

    if(strstr(ifName, IF_NAME_ETH))
    {
        macAddressType = MAC_ADDRESS_ETH;
    }
    else if(strstr(ifName, IF_NAME_USB))
    {
        macAddressType = MAC_ADDRESS_USB;
    }
    else if(strstr(ifName, IF_NAME_WLAN))
    {
        macAddressType = MAC_ADDRESS_WLAN;
    }
    else if(strstr(ifName, IF_NAME_MOCA))
    {
        macAddressType = MAC_ADDRESS_MOCA;
    }
    else if(strstr(ifName, IF_NAME_ATM))
    {
        macAddressType = MAC_ADDRESS_ATM;
    }
    else if(strstr(ifName, IF_NAME_PTM))
    {
        macAddressType = MAC_ADDRESS_PTM;
    }
    else if(strstr(ifName, IF_NAME_GPON) || strstr(ifName, IF_NAME_VEIP))
    {
        macAddressType = MAC_ADDRESS_GPON;
    }
    else if(strstr(ifName, IF_NAME_EPON))
    {
        macAddressType = MAC_ADDRESS_EPON;
    }

    return macAddressType;
}

static inline void kerSysMacAddressNotify(unsigned char *pucaMacAddr, MAC_ADDRESS_OPERATION op)
{
    if(kerSysMacAddressNotifyHook)
    {
        kerSysMacAddressNotifyHook(pucaMacAddr, op);
    }
}

int kerSysMacAddressNotifyBind(kerSysMacAddressNotifyHook_t hook)
{
    int nRet = 0;

    if(hook && kerSysMacAddressNotifyHook)
    {
        printk("ERROR: kerSysMacAddressNotifyHook already registered! <0x%08lX>\n",
               (unsigned long)kerSysMacAddressNotifyHook);
        nRet = -EINVAL;
    }
    else
    {
        kerSysMacAddressNotifyHook = hook;
    }

    return nRet;
}

int kerSysGetMacAddress( unsigned char *pucaMacAddr, unsigned long ulId )
{
    const unsigned long constMacAddrIncIndex = 3;
    int nRet = 0;
    PMAC_ADDR_INFO pMai = NULL;
    PMAC_ADDR_INFO pMaiFreeNoId = NULL;
    PMAC_ADDR_INFO pMaiFreeId = NULL;
    unsigned long i = 0, ulIdxNoId = 0, ulIdxId = 0, baseMacAddr = 0;

    mutex_lock(&macAddrMutex);

    /* baseMacAddr = last 3 bytes of the base MAC address treated as a 24 bit integer */
    memcpy((unsigned char *) &baseMacAddr,
        &g_pMacInfo->ucaBaseMacAddr[constMacAddrIncIndex],
        NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex);
    baseMacAddr >>= 8;

    for( i = 0, pMai = g_pMacInfo->MacAddrs; i < g_pMacInfo->ulNumMacAddrs;
        i++, pMai++ )
    {
        if( ulId == pMai->ulId || ulId == MAC_ADDRESS_ANY )
        {
            /* This MAC address has been used by the caller in the past. */
            baseMacAddr = (baseMacAddr + i) << 8;
            memcpy( pucaMacAddr, g_pMacInfo->ucaBaseMacAddr,
                constMacAddrIncIndex);
            memcpy( pucaMacAddr + constMacAddrIncIndex, (unsigned char *)
                &baseMacAddr, NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex );
            pMai->chInUse = 1;
            pMaiFreeNoId = pMaiFreeId = NULL;
            break;
        }
        else
            if( pMai->chInUse == 0 )
            {
                if( pMai->ulId == 0 && pMaiFreeNoId == NULL )
                {
                    /* This is an available MAC address that has never been
                    * used.
                    */
                    pMaiFreeNoId = pMai;
                    ulIdxNoId = i;
                }
                else
                    if( pMai->ulId != 0 && pMaiFreeId == NULL )
                    {
                        /* This is an available MAC address that has been used
                        * before.  Use addresses that have never been used
                        * first, before using this one.
                        */
                        pMaiFreeId = pMai;
                        ulIdxId = i;
                    }
            }
    }

    if( pMaiFreeNoId || pMaiFreeId )
    {
        /* An available MAC address was found. */
        memcpy(pucaMacAddr, g_pMacInfo->ucaBaseMacAddr,NVRAM_MAC_ADDRESS_LEN);
        if( pMaiFreeNoId )
        {
            baseMacAddr = (baseMacAddr + ulIdxNoId) << 8;
            memcpy( pucaMacAddr, g_pMacInfo->ucaBaseMacAddr,
                constMacAddrIncIndex);
            memcpy( pucaMacAddr + constMacAddrIncIndex, (unsigned char *)
                &baseMacAddr, NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex );
            pMaiFreeNoId->ulId = ulId;
            pMaiFreeNoId->chInUse = 1;
        }
        else
        {
            baseMacAddr = (baseMacAddr + ulIdxId) << 8;
            memcpy( pucaMacAddr, g_pMacInfo->ucaBaseMacAddr,
                constMacAddrIncIndex);
            memcpy( pucaMacAddr + constMacAddrIncIndex, (unsigned char *)
                &baseMacAddr, NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex );
            pMaiFreeId->ulId = ulId;
            pMaiFreeId->chInUse = 1;
        }

        kerSysMacAddressNotify(pucaMacAddr, MAC_ADDRESS_OP_GET);
    }
    else
        if( i == g_pMacInfo->ulNumMacAddrs )
            nRet = -EADDRNOTAVAIL;

    mutex_unlock(&macAddrMutex);

    return( nRet );
} /* kerSysGetMacAddr */

/* Allocates requested number of consecutive MAC addresses */
int kerSysGetMacAddresses( unsigned char *pucaMacAddr, unsigned int num_addresses, unsigned long ulId )
{
    const unsigned long constMacAddrIncIndex = 3;
    int nRet = 0;
    PMAC_ADDR_INFO pMai = NULL;
    PMAC_ADDR_INFO pMaiFreeId = NULL, pMaiFreeIdTemp;
    unsigned long i = 0, j = 0, ulIdxId = 0, baseMacAddr = 0;

    mutex_lock(&macAddrMutex);

    /* baseMacAddr = last 3 bytes of the base MAC address treated as a 24 bit integer */
    memcpy((unsigned char *) &baseMacAddr,
        &g_pMacInfo->ucaBaseMacAddr[constMacAddrIncIndex],
        NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex);
    baseMacAddr >>= 8;

#if defined(CONFIG_BCM96828)
    /*As epon mac should not be dynamicly changed, always use last 1(SLLID) or 8(MLLID) mac address(es)*/
    if (ulId == MAC_ADDRESS_EPONONU)
    {
        i = g_pMacInfo->ulNumMacAddrs - num_addresses; 

        for (j = 0, pMai = &g_pMacInfo->MacAddrs[i]; j < num_addresses; j++) {
            pMaiFreeIdTemp = pMai + j;
            if (pMaiFreeIdTemp->chInUse != 0 && pMaiFreeIdTemp->ulId != MAC_ADDRESS_EPONONU) {
                printk("kerSysGetMacAddresses: epon mac address allocate failed, g_pMacInfo[%ld] reserved by 0x%lx\n", i+j, pMaiFreeIdTemp->ulId);	
                break;
            }
        }
		
        if (j >= num_addresses) {
            pMaiFreeId = pMai;
            ulIdxId = i;
        }
    }
    else
#endif	
    {
        for( i = 0, pMai = g_pMacInfo->MacAddrs; i < g_pMacInfo->ulNumMacAddrs;
            i++, pMai++ )
        {
            if( ulId == pMai->ulId || ulId == MAC_ADDRESS_ANY )
            {
                /* This MAC address has been used by the caller in the past. */
                baseMacAddr = (baseMacAddr + i) << 8;
                memcpy( pucaMacAddr, g_pMacInfo->ucaBaseMacAddr,
                    constMacAddrIncIndex);
                memcpy( pucaMacAddr + constMacAddrIncIndex, (unsigned char *)
                    &baseMacAddr, NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex );
                pMai->chInUse = 1;
                pMaiFreeId = NULL;
                break;
            } else if( pMai->chInUse == 0 ) {
                /* check if it there are num_addresses to be checked starting from found MAC address */
                if ((i + num_addresses) >= g_pMacInfo->ulNumMacAddrs) {
                    nRet = -EADDRNOTAVAIL;
                    break;
                }
    
                for (j = 0; j < num_addresses; j++) {
                    pMaiFreeIdTemp = pMai + j;
                    if (pMaiFreeIdTemp->chInUse != 0) {
                        break;
                    }
                }
                if (j >= num_addresses) {
                    pMaiFreeId = pMai;
                    ulIdxId = i;
                    break;
                }
            }
        }
    }

    if(pMaiFreeId )
    {
        /* An available MAC address was found. */
        memcpy(pucaMacAddr, g_pMacInfo->ucaBaseMacAddr,NVRAM_MAC_ADDRESS_LEN);
        baseMacAddr = (baseMacAddr + ulIdxId) << 8;
        memcpy( pucaMacAddr, g_pMacInfo->ucaBaseMacAddr,
                constMacAddrIncIndex);
        memcpy( pucaMacAddr + constMacAddrIncIndex, (unsigned char *)
                &baseMacAddr, NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex );
        pMaiFreeIdTemp = pMai;
        for (j = 0; j < num_addresses; j++) {
            pMaiFreeIdTemp->ulId = ulId;
            pMaiFreeIdTemp->chInUse = 1;
            pMaiFreeIdTemp++;
        }
    }
    else {
        nRet = -EADDRNOTAVAIL;
    }

    mutex_unlock(&macAddrMutex);

    return( nRet );
} /* kerSysGetMacAddr */

int kerSysReleaseMacAddresses( unsigned char *pucaMacAddr, unsigned int num_addresses )
{
    const unsigned long constMacAddrIncIndex = 3;
    int i, nRet = -EINVAL;
    unsigned long ulIdx = 0;
    unsigned long baseMacAddr = 0;
    unsigned long relMacAddr = 0;

    mutex_lock(&macAddrMutex);

    /* baseMacAddr = last 3 bytes of the base MAC address treated as a 24 bit integer */
    memcpy((unsigned char *) &baseMacAddr,
        &g_pMacInfo->ucaBaseMacAddr[constMacAddrIncIndex],
        NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex);
    baseMacAddr >>= 8;

    /* Get last 3 bytes of MAC address to release. */
    memcpy((unsigned char *) &relMacAddr, &pucaMacAddr[constMacAddrIncIndex],
        NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex);
    relMacAddr >>= 8;

    ulIdx = relMacAddr - baseMacAddr;

    if( ulIdx < g_pMacInfo->ulNumMacAddrs )
    {
        for(i=0; i<num_addresses; i++) {
            if ((ulIdx + i) < g_pMacInfo->ulNumMacAddrs) {
                PMAC_ADDR_INFO pMai = &g_pMacInfo->MacAddrs[ulIdx + i];
                if( pMai->chInUse == 1 )
                {
                    pMai->chInUse = 0;
                    nRet = 0;
                }
            } else {
                printk("Request to release %d addresses failed as "
                    " the one or more of the addresses, starting from"
                    " %dth address from given address, requested for release"
                    " is not in the list of available MAC addresses \n", num_addresses, i);
                break;
            }
        }
    }

    mutex_unlock(&macAddrMutex);

    return( nRet );
} /* kerSysReleaseMacAddr */


int kerSysReleaseMacAddress( unsigned char *pucaMacAddr )
{
    const unsigned long constMacAddrIncIndex = 3;
    int nRet = -EINVAL;
    unsigned long ulIdx = 0;
    unsigned long baseMacAddr = 0;
    unsigned long relMacAddr = 0;

    mutex_lock(&macAddrMutex);

    /* baseMacAddr = last 3 bytes of the base MAC address treated as a 24 bit integer */
    memcpy((unsigned char *) &baseMacAddr,
        &g_pMacInfo->ucaBaseMacAddr[constMacAddrIncIndex],
        NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex);
    baseMacAddr >>= 8;

    /* Get last 3 bytes of MAC address to release. */
    memcpy((unsigned char *) &relMacAddr, &pucaMacAddr[constMacAddrIncIndex],
        NVRAM_MAC_ADDRESS_LEN - constMacAddrIncIndex);
    relMacAddr >>= 8;

    ulIdx = relMacAddr - baseMacAddr;

    if( ulIdx < g_pMacInfo->ulNumMacAddrs )
    {
        PMAC_ADDR_INFO pMai = &g_pMacInfo->MacAddrs[ulIdx];
        if( pMai->chInUse == 1 )
        {
            kerSysMacAddressNotify(pucaMacAddr, MAC_ADDRESS_OP_RELEASE);

            pMai->chInUse = 0;
            nRet = 0;
        }
    }

    mutex_unlock(&macAddrMutex);

    return( nRet );
} /* kerSysReleaseMacAddr */


void kerSysGetGponSerialNumber( unsigned char *pGponSerialNumber )
{
    strcpy( pGponSerialNumber, g_pGponInfo->gponSerialNumber );
}


void kerSysGetGponPassword( unsigned char *pGponPassword )
{
    strcpy( pGponPassword, g_pGponInfo->gponPassword );
}

int kerSysGetSdramSize( void )
{
    return( (int) g_ulSdramSize );
} /* kerSysGetSdramSize */


#if defined(CONFIG_BCM96368)
/*
 * This function returns:
 * MEMC_32BIT_BUS for 32-bit SDRAM
 * MEMC_16BIT_BUS for 16-bit SDRAM
 */
unsigned int kerSysGetSdramWidth( void )
{
    return (unsigned int)(g_ulSdramWidth);
} /* kerSysGetSdramWidth */
#endif


/*Read Wlan Params data from CFE */
int kerSysGetWlanSromParams( unsigned char *wlanParams, unsigned short len)
{
    NVRAM_DATA *pNvramData;

    if (NULL == (pNvramData = readNvramData()))
    {
        printk("kerSysGetWlanSromParams: could not read nvram data\n");
        return -1;
    }

    memcpy( wlanParams,
           (char *)pNvramData + ((size_t) &((NVRAM_DATA *)0)->wlanParams),
            len );
    kfree(pNvramData);

    return 0;
}

/*Read Wlan Params data from CFE */
int kerSysGetAfeId( unsigned long *afeId )
{
    NVRAM_DATA *pNvramData;

    if (NULL == (pNvramData = readNvramData()))
    {
        printk("kerSysGetAfeId: could not read nvram data\n");
        return -1;
    }

    afeId [0] = pNvramData->afeId[0];
    afeId [1] = pNvramData->afeId[1];
    kfree(pNvramData);

    return 0;
}

void kerSysLedCtrl(BOARD_LED_NAME ledName, BOARD_LED_STATE ledState)
{
    if (g_ledInitialized)
        boardLedCtrl(ledName, ledState);
}

/*functionto receive message from usersapce
 * Currently we dont expect any messages fromm userspace
 */
void kerSysRecvFrmMonitorTask(struct sk_buff *skb)
{

   /*process the message here*/
   printk(KERN_WARNING "unexpected skb received at %s \n",__FUNCTION__);
   kfree_skb(skb);
   return;
}

void kerSysInitMonitorSocket( void )
{
   g_monitor_nl_sk = netlink_kernel_create(&init_net, NETLINK_BRCM_MONITOR, 0, kerSysRecvFrmMonitorTask, NULL, THIS_MODULE);

   if(!g_monitor_nl_sk)
   {
      printk(KERN_ERR "Failed to create a netlink socket for monitor\n");
      return;
   }

}


#if 1 //#if defined(ATI_BSP_PERSONALITY)
typedef struct sSavedMsgs
{
  int msgType;
  char *msgData;
  int msgDataLen;
  struct sSavedMsgs *pNext;
} tSavedMsgs;
tSavedMsgs *pHeadSavedMsgs=NULL;
tSavedMsgs *pTailSavedMsgs=NULL;
#endif

static inline void kerSysSendtoMonitorTaskHelper(int msgType, char *msgData, int msgDataLen)
{
  struct sk_buff *skb =  NULL;
  struct nlmsghdr *nl_msgHdr = NULL;
  unsigned int payloadLen =sizeof(struct nlmsghdr);

  if(msgData && (msgDataLen > MAX_PAYLOAD_LEN))
  {
     printk(KERN_ERR "invalid message len in %s\n",__FUNCTION__);
     return;
  } 

  payloadLen += msgDataLen;
  payloadLen = NLMSG_SPACE(payloadLen);

  /*Alloc skb ,this check helps to call the fucntion from interrupt context */

  if(in_atomic())
  {
     skb = alloc_skb(payloadLen, GFP_ATOMIC);
  }
  else
  {
     skb = alloc_skb(payloadLen, GFP_KERNEL);
  }

  if(!skb)
  {
     printk(KERN_ERR "failed to alloc skb in %s\n",__FUNCTION__);
     return;
  }

  nl_msgHdr = (struct nlmsghdr *)skb->data;
  nl_msgHdr->nlmsg_type = msgType;
  nl_msgHdr->nlmsg_pid=0;/*from kernel */
  nl_msgHdr->nlmsg_len = payloadLen;
  nl_msgHdr->nlmsg_flags =0;

  if(msgData)
  {
     memcpy(NLMSG_DATA(nl_msgHdr),msgData,msgDataLen);
  }      

  NETLINK_CB(skb).pid = 0; /*from kernel */

  skb->len = payloadLen; 

  netlink_unicast(g_monitor_nl_sk, skb, g_monitor_nl_pid, MSG_DONTWAIT);
  return;
}

void kerSysSendtoMonitorTask(int msgType, char *msgData, int msgDataLen)
{
   if(!g_monitor_nl_pid)
   {
#if 0 //#if defined(ATI_BSP_PERSONALITY)
      printk(KERN_INFO "message received before monitor task is initialized %s \n",__FUNCTION__);
   }
#else
     printk(KERN_INFO "Qing: (0x%x) message\n",
            msgType);
      if (!pHeadSavedMsgs)
      {
        if(in_atomic())
          pHeadSavedMsgs = (tSavedMsgs *)kmalloc(sizeof(tSavedMsgs), GFP_ATOMIC);
        else
          pHeadSavedMsgs = (tSavedMsgs *)kmalloc(sizeof(tSavedMsgs), GFP_KERNEL);
        pTailSavedMsgs = pHeadSavedMsgs;
        pHeadSavedMsgs->pNext = NULL;
      }
      else
      {
        if(in_atomic())
          pTailSavedMsgs->pNext = (tSavedMsgs *)kmalloc(sizeof(tSavedMsgs), GFP_ATOMIC);
        else
          pTailSavedMsgs->pNext = (tSavedMsgs *)kmalloc(sizeof(tSavedMsgs), GFP_KERNEL);
        pTailSavedMsgs = pTailSavedMsgs->pNext;
        pTailSavedMsgs->pNext = NULL;
      }

      if (pTailSavedMsgs)
      {
        pTailSavedMsgs->msgType = msgType;
        if(msgDataLen && msgData)
        {
          // Allocate space
          if(in_atomic())
            pTailSavedMsgs->msgData = (char *)kmalloc(msgDataLen, GFP_ATOMIC);
          else
            pTailSavedMsgs->msgData = (char *)kmalloc(msgDataLen, GFP_KERNEL);
          pTailSavedMsgs->msgDataLen = msgDataLen;
          memcpy(pTailSavedMsgs->msgData, msgData, msgDataLen);
        }
        else
        {
          pTailSavedMsgs->msgData = NULL;
          pTailSavedMsgs->msgDataLen = 0;
        }
      }
      return;
   }

   // Clear any saved messages 
   while (pHeadSavedMsgs)
   {
     pTailSavedMsgs=pHeadSavedMsgs;
     kerSysSendtoMonitorTaskHelper(pHeadSavedMsgs->msgType, 
                                   pHeadSavedMsgs->msgData, 
                                   pHeadSavedMsgs->msgDataLen);
     pHeadSavedMsgs=pHeadSavedMsgs->pNext;
     if (pTailSavedMsgs->msgData)
     {
       kfree(pTailSavedMsgs->msgData);
     }
     kfree(pTailSavedMsgs);
   }

   // code moved to:
   kerSysSendtoMonitorTaskHelper(msgType, msgData, msgDataLen);
#endif
}

void kerSysCleanupMonitorSocket(void)
{
   g_monitor_nl_pid = 0 ;
   sock_release(g_monitor_nl_sk->sk_socket);
}

// Must be called with flashImageMutex held
static PFILE_TAG getTagFromPartition(int imageNumber)
{
#if !defined(ATI_BSP_PERSONALITY)   
    // Define space for file tag structures for two partitions.  Make them static
    // so caller does not have to worry about allocation/deallocation.
    // Make sure they're big enough for the file tags plus an block number
    // (an integer) appended.
    static unsigned char sectAddr1[sizeof(FILE_TAG) + sizeof(int)];
    static unsigned char sectAddr2[sizeof(FILE_TAG) + sizeof(int)];

    int blk = 0;
    UINT32 crc;
    PFILE_TAG pTag = NULL;
    unsigned char *pBase = flash_get_memptr(0);
    unsigned char *pSectAddr = NULL;

    unsigned  int reserverdBytersAuxfs = flash_get_reserved_bytes_auxfs();

    /* The image tag for the first image is always after the boot loader.
     * The image tag for the second image, if it exists, is at one half
     * of the flash size.
     */
    if( imageNumber == 1 )
    {
        // Get the flash info and block number for parition 1 at the base of the flash
        FLASH_ADDR_INFO flash_info;

        kerSysFlashAddrInfoGet(&flash_info);
        blk = flash_get_blk((int)(pBase+flash_info.flash_rootfs_start_offset));
        pSectAddr = sectAddr1;
    }
    else if( imageNumber == 2 )
    {
        // Get block number for partition2 at middle of the device (not counting space for aux
        // file system).
        blk = flash_get_blk((int) (pBase + ((flash_get_total_size()-reserverdBytersAuxfs) / 2)));
        pSectAddr = sectAddr2;
    }
    
    // Now that you have a block number, use it to read in the file tag
    if( blk )
    {
        int *pn;    // Use to append block number at back of file tag
        
        // Clear out space for file tag structures
        memset(pSectAddr, 0x00, sizeof(FILE_TAG));
        
        // Get file tag
        flash_read_buf((unsigned short) blk, 0, pSectAddr, sizeof(FILE_TAG));
        
        // Figure out CRC of file tag so we can check it below
        crc = CRC32_INIT_VALUE;
        crc = getCrc32(pSectAddr, (UINT32)TAG_LEN-TOKEN_LEN, crc);
        
        // Get ready to return file tag pointer
        pTag = (PFILE_TAG) pSectAddr;
        
        // Append block number after file tag
        pn = (int *) (pTag + 1);
        *pn = blk;
        
        // One final check - if the file tag CRC is not OK, return NULL instead
        if (crc != (UINT32)(*(UINT32*)(pTag->tagValidationToken)))
            pTag = NULL;
    }
    
    // All done - return file tag pointer
#else
   int blk = 0;
   UINT32 crc;
   PFILE_TAG pTag = NULL;
   unsigned char *pSectAddr = NULL;
   static unsigned char sectAddr1[sizeof(FILE_TAG) + sizeof(int)];
   static unsigned char sectAddr2[sizeof(FILE_TAG) + sizeof(int)];

   if (imageNumber == 1)
   {
      char *tagaddr;
      if ((tagaddr = strstr(atiNvramData.bcmNvRam.szBootline, "AtiImage1TagAddr")))
      {
         pSectAddr = (void *)simple_strtoul(strstr(tagaddr, "=")+1, NULL, 16);
         pTag = (PFILE_TAG)sectAddr1;
      }
   }
   else if (imageNumber == 2)
   {
      char *tagaddr;
      if ((tagaddr = strstr(atiNvramData.bcmNvRam.szBootline, "AtiImage2TagAddr")))
      {
         pSectAddr = (void *)simple_strtoul(strstr(tagaddr, "=")+1, NULL, 16);
         pTag = (PFILE_TAG)sectAddr2;
      }
   }

   blk = flash_get_blk((unsigned int)pSectAddr);
   // Now that you have a block number, use it to read in the file tag
   if ( blk )
   {
      int *pn;    // Use to append block number at back of file tag

      // Clear out space for file tag structures
      memset(pTag, 0x00, sizeof(FILE_TAG));

      // Get file tag
      flash_read_buf((unsigned short) blk, 0, (char*)pTag, sizeof(FILE_TAG));

      // Figure out CRC of file tag so we can check it below
      crc = CRC32_INIT_VALUE;
      crc = getCrc32((char*)pTag, (UINT32)TAG_LEN-TOKEN_LEN, crc);

      // Append block number after file tag
      pn = (int *) (pTag + 1);
      *pn = blk;

      // One final check - if the file tag CRC is not OK, return NULL instead
      if (crc != (UINT32)(*(UINT32*)(pTag->tagValidationToken)))
      {
         pTag = NULL;
      }
   }

#endif
    return( pTag );
}


#if !defined(ATI_PRODUCT_CONFIG)
// must be called with flashImageMutex held
static int getPartitionFromTag( PFILE_TAG pTag )
{
    int ret = 0;

    if( pTag )
    {
#if !defined(ATI_BSP_PERSONALITY)
        PFILE_TAG pTag1 = getTagFromPartition(1);
        PFILE_TAG pTag2 = getTagFromPartition(2);
        int sequence = simple_strtoul(pTag->imageSequence,  NULL, 10);
        int sequence1 = (pTag1) ? simple_strtoul(pTag1->imageSequence, NULL, 10)
            : -1;
        int sequence2 = (pTag2) ? simple_strtoul(pTag2->imageSequence, NULL, 10)
            : -1;
#else
        FILE_TAG  Tag1,Tag2,currentTag;
        PFILE_TAG pTag1 = kerSysImageTagCpy(1, &Tag1);
        PFILE_TAG pTag2 = kerSysImageTagCpy(2, &Tag2);
        int sequence, sequence1, sequence2;
        if (((unsigned long)pTag & FLASH_BASE) == FLASH_BASE)
          kerSysReadFromFlash((void *)&currentTag, (unsigned long)pTag, sizeof(FILE_TAG));
        else
          memcpy(&currentTag, pTag, sizeof(FILE_TAG));
        sequence = simple_strtoul((const char *)currentTag.imageSequence,  NULL, 10);
        sequence1 = (pTag1) ? simple_strtoul(Tag1.imageSequence, NULL, 10)
            : -1;
        sequence2 = (pTag2) ? simple_strtoul(Tag2.imageSequence, NULL, 10)
            : -1;
#endif

        if( pTag1 && sequence == sequence1 )
            ret = 1;
        else
            if( pTag2 && sequence == sequence2 )
                ret = 2;
    }

    return( ret );
}
#endif

// must be called with flashImageMutex held
static PFILE_TAG getBootImageTag(void)
{
    static int displayFsAddr = 1;
    PFILE_TAG pTag = NULL;
#if !defined(ATI_BSP_PERSONALITY)
    PFILE_TAG pTag1 = getTagFromPartition(1);
    PFILE_TAG pTag2 = getTagFromPartition(2);
#endif

#if defined(ATI_BSP_PERSONALITY)
    char *tagaddr;
    // The AtiBcmRamTagAddr over rides the flash based AtiBcmTagAddr - order here is important
    if ( (tagaddr = strstr(atiNvramData.bcmNvRam.szBootline, "AtiBcmRamTagAddr")) || 
         (tagaddr = strstr(atiNvramData.bcmNvRam.szBootline, "AtiBcmTagAddr")) )
    {
      pTag = (void *)simple_strtoul(strstr(tagaddr, "=")+1, NULL, 16);
    }
#else
    BCM_ASSERT_HAS_MUTEX_C(&flashImageMutex);

    if( pTag1 && pTag2 )
    {
        /* Two images are flashed. */
        int sequence1 = simple_strtoul(pTag1->imageSequence, NULL, 10);
        int sequence2 = simple_strtoul(pTag2->imageSequence, NULL, 10);
        int imgid = 0;

        kerSysBlParmsGetInt(BOOTED_IMAGE_ID_NAME, &imgid);
        if( imgid == BOOTED_OLD_IMAGE )
            pTag = (sequence2 < sequence1) ? pTag2 : pTag1;
        else
            pTag = (sequence2 > sequence1) ? pTag2 : pTag1;
    }
    else
        /* One image is flashed. */
        pTag = (pTag2) ? pTag2 : pTag1;
#endif

    if( pTag && displayFsAddr )
    {
        displayFsAddr = 0;
#if !defined(ATI_BSP_PERSONALITY)
        printk("File system address: 0x%8.8lx\n",
            simple_strtoul(pTag->rootfsAddress, NULL, 10) + BOOT_OFFSET);
#endif
    }

    return( pTag );
}

// Must be called with flashImageMutex held
static void UpdateImageSequenceNumber( unsigned char *imageSequence )
{
    int newImageSequence = 0;
#if !defined(ATI_BSP_PERSONALITY)
    PFILE_TAG pTag = getTagFromPartition(1);
#else
    FILE_TAG Tag;
    PFILE_TAG pTag = kerSysImageTagCpy(1, &Tag);
#endif

    if( pTag )
#if !defined(ATI_BSP_PERSONALITY)
        newImageSequence = simple_strtoul(pTag->imageSequence, NULL, 10);

#else
        newImageSequence = simple_strtoul(Tag.imageSequence, NULL, 10);
#endif
#if !defined(ATI_BSP_PERSONALITY)
    pTag = getTagFromPartition(2);
    if(pTag && simple_strtoul(pTag->imageSequence, NULL, 10) > newImageSequence)
        newImageSequence = simple_strtoul(pTag->imageSequence, NULL, 10);
#else
    pTag = kerSysImageTagCpy(2, &Tag);
    if(pTag && simple_strtoul(Tag.imageSequence, NULL, 10) > newImageSequence)
        newImageSequence = simple_strtoul(Tag.imageSequence, NULL, 10);
#endif

    newImageSequence++;
    sprintf(imageSequence, "%d", newImageSequence);
}

/* Must be called with flashImageMutex held */
static int flashFsKernelImage( unsigned char *imagePtr, int imageLen,
    int flashPartition, int *numPartitions )
{

    int status = 0;
    PFILE_TAG pTag = (PFILE_TAG) imagePtr;
    int rootfsAddr = simple_strtoul(pTag->rootfsAddress, NULL, 10);
    int kernelAddr = simple_strtoul(pTag->kernelAddress, NULL, 10);
#if !defined(ATI_PRODUCT_CONFIG)
    char *p;
#endif
    char *tagFs = imagePtr;
#if !defined(ATI_PRODUCT_CONFIG)
    unsigned int baseAddr = (unsigned int) flash_get_memptr(0);
    unsigned int totalSize = (unsigned int) flash_get_total_size();
    unsigned int reservedBytesAtEnd;
    unsigned int reserverdBytersAuxfs;
    unsigned int availableSizeOneImg;
    unsigned int reserveForTwoImages;
    unsigned int availableSizeTwoImgs;
    unsigned int newImgSize = simple_strtoul(pTag->rootfsLen, NULL, 10) +
        simple_strtoul(pTag->kernelLen, NULL, 10);
    PFILE_TAG pCurTag = getBootImageTag();
    int nCurPartition = getPartitionFromTag( pCurTag );
    int should_yield =
        (flashPartition == 0 || flashPartition == nCurPartition) ? 0 : 1;
#else
    int should_yield = 1; // We need to yeild during flash will also unblock IRQs
    unsigned int availableSizeTwoImgs = 0;
    unsigned int newImgSize = simple_strtoul(pTag->rootfsLen, NULL, 10) +
        simple_strtoul(pTag->kernelLen, NULL, 10);
//    PFILE_TAG pCurTag = getBootImageTag();
#endif
    UINT32 crc;
#if !defined(ATI_PRODUCT_CONFIG)
    unsigned int curImgSize = 0;
    unsigned int rootfsOffset = (unsigned int) rootfsAddr - IMAGE_BASE - TAG_LEN;
#endif
    FLASH_ADDR_INFO flash_info;
    NVRAM_DATA *pNvramData;
    BCM_ASSERT_HAS_MUTEX_C(&flashImageMutex);

    if (NULL == (pNvramData = readNvramData()))
    {
        return -ENOMEM;
    }

    kerSysFlashAddrInfoGet(&flash_info);

#if !defined(ATI_PRODUCT_CONFIG)
    if( rootfsOffset < flash_info.flash_rootfs_start_offset )
    {
        // Increase rootfs and kernel addresses by the difference between
        // rootfs offset and what it needs to be.
        rootfsAddr += flash_info.flash_rootfs_start_offset - rootfsOffset;
        kernelAddr += flash_info.flash_rootfs_start_offset - rootfsOffset;
        sprintf(pTag->rootfsAddress,"%lu", (unsigned long) rootfsAddr);
        sprintf(pTag->kernelAddress,"%lu", (unsigned long) kernelAddr);
        crc = CRC32_INIT_VALUE;
        crc = getCrc32((unsigned char *)pTag, (UINT32)TAG_LEN-TOKEN_LEN, crc);
        *(unsigned long *) &pTag->tagValidationToken[0] = crc;
    }

    rootfsAddr += BOOT_OFFSET;
    kernelAddr += BOOT_OFFSET;

    reservedBytesAtEnd = flash_get_reserved_bytes_at_end(&flash_info);
    reserverdBytersAuxfs = flash_get_reserved_bytes_auxfs();
    availableSizeOneImg = totalSize - ((unsigned int) rootfsAddr - baseAddr) -
        reservedBytesAtEnd- reserverdBytersAuxfs;  
        
    reserveForTwoImages =
        (flash_info.flash_rootfs_start_offset > reservedBytesAtEnd)
        ? flash_info.flash_rootfs_start_offset : reservedBytesAtEnd;
    availableSizeTwoImgs = ((totalSize-reserverdBytersAuxfs)/ 2) - reserveForTwoImages;

    printk("availableSizeOneImage=%dKB availableSizeTwoImgs=%dKB reserverdBytersAuxfs=%dKB reserve=%dKB\n",
        availableSizeOneImg/1024, availableSizeTwoImgs/1024, reserverdBytersAuxfs/1024, reserveForTwoImages/1024);
	   
    if( pCurTag )
    {
        curImgSize = simple_strtoul(pCurTag->rootfsLen, NULL, 10) +
            simple_strtoul(pCurTag->kernelLen, NULL, 10);
    }

    if( newImgSize > availableSizeOneImg)
    {
        printk("Illegal image size %d.  Image size must not be greater "
            "than %d.\n", newImgSize, availableSizeOneImg);
        kfree(pNvramData);
        return -1;
    }

    *numPartitions = (curImgSize <= availableSizeTwoImgs &&
         newImgSize <= availableSizeTwoImgs &&
         flashPartition != nCurPartition) ? 2 : 1;

    // If the current image fits in half the flash space and the new
    // image to flash also fits in half the flash space, then flash it
    // in the partition that is not currently being used to boot from.
    if( curImgSize <= availableSizeTwoImgs &&
        newImgSize <= availableSizeTwoImgs &&
        ((nCurPartition == 1 && flashPartition != 1) || flashPartition == 2) )
    {
    // Update rootfsAddr to point to the second boot partition.
		int offset = ((totalSize-reserverdBytersAuxfs) / 2) + TAG_LEN;       
        sprintf(((PFILE_TAG) tagFs)->kernelAddress, "%lu",
            (unsigned long) IMAGE_BASE + offset + (kernelAddr - rootfsAddr));
        kernelAddr = baseAddr + offset + (kernelAddr - rootfsAddr);

        sprintf(((PFILE_TAG) tagFs)->rootfsAddress, "%lu",
            (unsigned long) IMAGE_BASE + offset);
        rootfsAddr = baseAddr + offset;
    }
#else
    *numPartitions =  2; // ATI boards will have 2 partitions
    //*ATI_PRODUCT_CONFIG*/
    printk("\n rootfsAddr = <0x%08x>", rootfsAddr);
    printk("\n kernelAddr = <0x%08x>", kernelAddr);
    printk("\n newImgSize = <%d>", newImgSize);
    printk("\n FLASH_BASE = <0x%08lx>", FLASH_BASE);
    printk("\n flash_info.flash_rootfs_start_offset = <0x%08lx>", flash_info.flash_rootfs_start_offset);
    printk("\n flash_info.flash_rootfs_image2_start_offset = <0x%08lx>", flash_info.flash_rootfs_image2_start_offset);
    printk("\n flash_info.flash_booted_image_start = <0x%08lx>", flash_info.flash_booted_image_start);

    /*IF booting from one image, flash the other one..*/
    if(flash_info.flash_booted_image_start == flash_info.flash_rootfs_start_offset)
    {
      int kernel_offset = kernelAddr - rootfsAddr;
      int meta_start_offset = ((int) flash_get_memptr(flash_info.flash_meta_start_blk)) - FLASH_BASE;

      availableSizeTwoImgs = meta_start_offset - flash_info.flash_rootfs_image2_start_offset;
      //printk("\n meta_start_offset <0x%8x> availableSizeTwoImages = <%d>", meta_start_offset, availableSizeTwoImgs);

      rootfsAddr = FLASH_BASE + flash_info.flash_rootfs_image2_start_offset + TAG_LEN;
      kernelAddr = rootfsAddr + kernel_offset;
    }
    else if(flash_info.flash_booted_image_start == flash_info.flash_rootfs_image2_start_offset)
    {
      int kernel_offset = kernelAddr - rootfsAddr;

      availableSizeTwoImgs = flash_info.flash_rootfs_image2_start_offset - flash_info.flash_rootfs_start_offset;

      rootfsAddr = FLASH_BASE + flash_info.flash_rootfs_start_offset + TAG_LEN;
      kernelAddr = rootfsAddr + kernel_offset;
    }

    //printk("\n availableSizeTwoImages = <%d>", availableSizeTwoImgs);
      /*Actual size that will be written is newImgsSize (rootFsSize) + TAG_LEN*/
    if( (newImgSize + TAG_LEN) > availableSizeTwoImgs)
    {
        printk("\nIllegal image size %d.  Image size must not be greater "
            "than %d.\n", newImgSize, availableSizeTwoImgs);
        kfree(pNvramData);
        return -1;
    }

    printk("\n FLASHFS rootfsAddr = <0x%08x>", rootfsAddr);
    printk("\n FLASHFS kernelAddr = <0x%08x>", kernelAddr);
#endif

    UpdateImageSequenceNumber( ((PFILE_TAG) tagFs)->imageSequence );
    crc = CRC32_INIT_VALUE;
    crc = getCrc32((unsigned char *)tagFs, (UINT32)TAG_LEN-TOKEN_LEN, crc);
    *(unsigned long *) &((PFILE_TAG) tagFs)->tagValidationToken[0] = crc;

    
    // Now, perform the actual flash write
    if( (status = kerSysBcmImageSet((rootfsAddr-TAG_LEN), tagFs,
        TAG_LEN + newImgSize, should_yield)) != 0 )
    {
        printk("Failed to flash root file system. Error: %d\n", status);
        kfree(pNvramData);
        return status;
    }
    
    // Free buffers
    kfree(pNvramData);
    return(status);
}

static int getImageVersion( int imageNumber, char *verStr, int verStrSize)
{
    int ret = 0; /* zero bytes copied to verStr so far */

    PFILE_TAG pTag = NULL;

    if( imageNumber == 1 )
        pTag = getTagFromPartition(1);
    else
        if( imageNumber == 2 )
            pTag = getTagFromPartition(2);

    if( pTag )
    {
        if( verStrSize > sizeof(pTag->imageVersion) )
            ret = sizeof(pTag->imageVersion);
        else
            ret = verStrSize;

        memcpy(verStr, pTag->imageVersion, ret);
    }

    return( ret );
}

PFILE_TAG kerSysUpdateTagSequenceNumber(int imageNumber)
{
    PFILE_TAG pTag = NULL;
    UINT32 crc;

    switch( imageNumber )
    {
    case 0:
        pTag = getBootImageTag();
        break;

    case 1:
        pTag = getTagFromPartition(1);
        break;

    case 2:
        pTag = getTagFromPartition(2);
        break;

    default:
        break;
    }

    if( pTag )
    {
        UpdateImageSequenceNumber( pTag->imageSequence );
        crc = CRC32_INIT_VALUE;
        crc = getCrc32((unsigned char *)pTag, (UINT32)TAG_LEN-TOKEN_LEN, crc);
        *(unsigned long *) &pTag->tagValidationToken[0] = crc;
    }

    return(pTag);
}

int kerSysGetSequenceNumber(int imageNumber)
{
    PFILE_TAG pTag = NULL;
    int seqNumber = -1;

    switch( imageNumber )
    {
    case 0:
        pTag = getBootImageTag();
        break;

    case 1:
        pTag = getTagFromPartition(1);
        break;

    case 2:
        pTag = getTagFromPartition(2);
        break;

    default:
        break;
    }

    if( pTag )
        seqNumber= simple_strtoul(pTag->imageSequence, NULL, 10);

    return(seqNumber);
}

static int getBootedValue(int getBootedPartition)
{
    int ret = -1;
    int imgId = -1;

    kerSysBlParmsGetInt(BOOTED_IMAGE_ID_NAME, &imgId);

    /* The boot loader parameter will only be "new image", "old image" or "only
     * image" in order to be compatible with non-OMCI image update. If the
     * booted partition is requested, convert this boot type to partition type.
     */
    if( imgId != -1 )
    {
        if( getBootedPartition )
        {
            /* Get booted partition. */
            int seq1 = kerSysGetSequenceNumber(1);
            int seq2 = kerSysGetSequenceNumber(2);

            switch( imgId )
            {
            case BOOTED_NEW_IMAGE:
                if( seq1 == -1 || seq2 > seq1 )
                    ret = BOOTED_PART2_IMAGE;
                else
                    if( seq2 == -1 || seq1 >= seq2 )
                        ret = BOOTED_PART1_IMAGE;
                break;

            case BOOTED_OLD_IMAGE:
                if( seq1 == -1 || seq2 < seq1 )
                    ret = BOOTED_PART2_IMAGE;
                else
                    if( seq2 == -1 || seq1 <= seq2 )
                        ret = BOOTED_PART1_IMAGE;
                break;

            case BOOTED_ONLY_IMAGE:
                ret = (seq1 == -1) ? BOOTED_PART2_IMAGE : BOOTED_PART1_IMAGE;
                break;

            default:
                break;
            }
        }
        else
            ret = imgId;
    }

    return( ret );
}


#if !defined(CONFIG_BRCM_IKOS)
PFILE_TAG kerSysImageTagGet(void)
{
    PFILE_TAG tag;

    mutex_lock(&flashImageMutex);
    tag = getBootImageTag();
    mutex_unlock(&flashImageMutex);

    return tag;
}
#else
PFILE_TAG kerSysImageTagGet(void)
{
    return( (PFILE_TAG) (FLASH_BASE + FLASH_LENGTH_BOOT_ROM));
}
#endif

#if defined(ATI_BSP_PERSONALITY)
PFILE_TAG kerSysImageTagCpy(int imageNumber, PFILE_TAG pTag)
{
    PFILE_TAG pTmpTag;
    if (!imageNumber)
      pTmpTag = getBootImageTag();
    else
      pTmpTag = getTagFromPartition(imageNumber);

    if (pTmpTag)
    {
      if (((unsigned long)pTmpTag & FLASH_BASE) == FLASH_BASE)
        kerSysReadFromFlash((void *)pTag, (unsigned long)pTmpTag, sizeof(FILE_TAG));
      else
        memcpy(pTag, pTmpTag, sizeof(FILE_TAG));
    }
    else
      memset(pTag, 0, sizeof(FILE_TAG));

    printk("Tag: %p File system offset: 0x%8.8lx\n", 
           pTmpTag, simple_strtoul(pTag->rootfsAddress, NULL, 10));
    return(pTmpTag);
}
#endif

/*
 * Common function used by BCM_IMAGE_CFE and BCM_IMAGE_WHOLE ioctls.
 * This function will acquire the flashImageMutex
 *
 * @return 0 on success, -1 on failure.
 */
static int commonImageWrite(int flash_start_addr, char *string, int size)
{
    NVRAM_DATA * pNvramDataOrig;
    NVRAM_DATA * pNvramDataNew=NULL;
    int ret;

    mutex_lock(&flashImageMutex);

    // Get a copy of the nvram before we do the image write operation
    if (NULL != (pNvramDataOrig = readNvramData()))
    {

        ret = kerSysBcmImageSet(flash_start_addr, string, size, 0);

        /*
         * After the image is written, check the nvram.
         * If nvram is bad, write back the original nvram.
         */
        pNvramDataNew = readNvramData();
        if ((0 != ret) ||
            (NULL == pNvramDataNew) ||
            (BpSetBoardId(pNvramDataNew->szBoardId) != BP_SUCCESS)
#if defined (CONFIG_BCM_ENDPOINT_MODULE)
            || (BpSetVoiceBoardId(pNvramDataNew->szVoiceBoardId) != BP_SUCCESS)
#endif
            )
        {
            // we expect this path to be taken.  When a CFE or whole image
            // is written, it typically does not have a valid nvram block
            // in the image.  We detect that condition here and restore
            // the previous nvram settings.  Don't print out warning here.
            writeNvramDataCrcLocked(pNvramDataOrig);

            // don't modify ret, it is return value from kerSysBcmImageSet
        }
    }
    else
    {
        ret = -1;
    }

    mutex_unlock(&flashImageMutex);

    if (pNvramDataOrig)
        kfree(pNvramDataOrig);
    if (pNvramDataNew)
        kfree(pNvramDataNew);

    return ret;
}


struct file_operations monitor_fops;

//********************************************************************************************
// misc. ioctl calls come to here. (flash, led, reset, kernel memory access, etc.)
//********************************************************************************************
static int board_ioctl( struct inode *inode, struct file *flip,
                       unsigned int command, unsigned long arg )
{
    int ret = 0;
    BOARD_IOCTL_PARMS ctrlParms;
    unsigned char ucaMacAddr[NVRAM_MAC_ADDRESS_LEN];

    switch (command) {
    case BOARD_IOCTL_FLASH_WRITE:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {

            switch (ctrlParms.action) {
            case SCRATCH_PAD:
                if (ctrlParms.offset == -1)
                    ret =  kerSysScratchPadClearAll();
                else
                    ret = kerSysScratchPadSet(ctrlParms.string, ctrlParms.buf, ctrlParms.offset);
                break;

            case PERSISTENT:
                ret = kerSysPersistentSet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                break;
#if defined(ATI_PRODUCT_CONFIG)
             case ATI_CONFIG_1:
             case ATI_CONFIG_2:
             case ATI_CONFIG_FACTORY_CUSTOMER:
                 ret = kerSysATIConfigSet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset, ctrlParms.action);
                 break;
#endif    /* ^ ATI_PRODUCT_CONFIG ^ */

            case BACKUP_PSI:
                ret = kerSysBackupPsiSet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                break;

            case SYSLOG:
                ret = kerSysSyslogSet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                break;

            case NVRAM:
            {
                NVRAM_DATA * pNvramData;

                /*
                 * Note: even though NVRAM access is protected by
                 * flashImageMutex at the kernel level, this protection will
                 * not work if two userspaces processes use ioctls to get
                 * NVRAM data, modify it, and then use this ioctl to write
                 * NVRAM data.  This seems like an unlikely scenario.
                 */
                mutex_lock(&flashImageMutex);
                if (NULL == (pNvramData = readNvramData()))
                {
                    mutex_unlock(&flashImageMutex);
                    return -ENOMEM;
                }
                if ( !strncmp(ctrlParms.string, "WLANDATA", 8 ) ) { //Wlan Data data
                    memset((char *)pNvramData + ((size_t) &((NVRAM_DATA *)0)->wlanParams),
                        0, sizeof(pNvramData->wlanParams) );
                    memcpy( (char *)pNvramData + ((size_t) &((NVRAM_DATA *)0)->wlanParams),
                        ctrlParms.string+8,
                        ctrlParms.strLen-8);
                    writeNvramDataCrcLocked(pNvramData);
                }
                else {
                    // assumes the user has calculated the crc in the nvram struct
                    ret = kerSysNvRamSet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                }
                mutex_unlock(&flashImageMutex);
                kfree(pNvramData);
                break;
            }

            case BCM_IMAGE_CFE:
                if( ctrlParms.strLen <= 0 || ctrlParms.strLen > FLASH_LENGTH_BOOT_ROM )
                {
                    printk("Illegal CFE size [%d]. Size allowed: [%d]\n",
                        ctrlParms.strLen, FLASH_LENGTH_BOOT_ROM);
                    ret = -1;
                    break;
                }

                ret = commonImageWrite(ctrlParms.offset + BOOT_OFFSET, ctrlParms.string, ctrlParms.strLen);

                break;

            case BCM_IMAGE_FS:
                {
                int numPartitions = 1;
#if !defined(ATI_BSP_PERSONALITY)
//Warning as error
                int noReboot = FLASH_IS_NO_REBOOT(ctrlParms.offset);
#endif
                int partition = FLASH_GET_PARTITION(ctrlParms.offset);

                mutex_lock(&flashImageMutex);
                ret = flashFsKernelImage(ctrlParms.string, ctrlParms.strLen,
                    partition, &numPartitions);
                mutex_unlock(&flashImageMutex);
#if !defined(ATI_BSP_PERSONALITY)
// We don't want to automatically reset the board.
                if(ret == 0 && (numPartitions == 1 || noReboot == 0))
                    resetPwrmgmtDdrMips();
#endif
                }
                break;

            case BCM_IMAGE_KERNEL:  // not used for now.
                break;

            case BCM_IMAGE_WHOLE:
                if(ctrlParms.strLen <= 0)
                {
                    printk("Illegal flash image size [%d].\n", ctrlParms.strLen);
                    ret = -1;
                    break;
                }

                if (ctrlParms.offset == 0)
                {
                    ctrlParms.offset = FLASH_BASE;
                }

                ret = commonImageWrite(ctrlParms.offset, ctrlParms.string, ctrlParms.strLen);

                if (ret == 0)
                {
                    resetPwrmgmtDdrMips();
                }
                else
                {
                    printk("flash of whole image failed, ret=%d\n", ret);
                }
                break;

            default:
                ret = -EINVAL;
                printk("flash_ioctl_command: invalid command %d\n", ctrlParms.action);
                break;
            }
            ctrlParms.result = ret;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_FLASH_READ:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            switch (ctrlParms.action) {
            case SCRATCH_PAD:
                ret = kerSysScratchPadGet(ctrlParms.string, ctrlParms.buf, ctrlParms.offset);
                break;

            case PERSISTENT:
                ret = kerSysPersistentGet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                break;
#if defined(ATI_PRODUCT_CONFIG)
             case ATI_CONFIG_1:
             case ATI_CONFIG_2:
             case ATI_CONFIG_FACTORY_CUSTOMER:
                ret = kerSysATIConfigGet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset, ctrlParms.action);
                break;
#endif    /* ^ ATI_PRODUCT_CONFIG ^ */

            case BACKUP_PSI:
                ret = kerSysBackupPsiGet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                break;

            case SYSLOG:
                ret = kerSysSyslogGet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                break;

            case NVRAM:
                kerSysNvRamGet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset);
                ret = 0;
                break;

            case FLASH_SIZE:
                ret = kerSysFlashSizeGet();
                break;

#if defined(ATI_PRODUCT_CONFIG)
            case BCM_IMAGE_FS_1:
            case BCM_IMAGE_FS_2:
                ret = kerSysATIImageGet(ctrlParms.string, ctrlParms.strLen, ctrlParms.offset, ctrlParms.action);
                break;
#endif    /* ^ ATI_PRODUCT_CONFIG ^ */

            default:
                ret = -EINVAL;
                printk("Not supported.  invalid command %d\n", ctrlParms.action);
                break;
            }
            ctrlParms.result = ret;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_FLASH_LIST:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            switch (ctrlParms.action) {
            case SCRATCH_PAD:
                ret = kerSysScratchPadList(ctrlParms.buf, ctrlParms.offset);
                break;

            default:
                ret = -EINVAL;
                printk("Not supported.  invalid command %d\n", ctrlParms.action);
                break;
            }
            ctrlParms.result = ret;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_DUMP_ADDR:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            dumpaddr( (unsigned char *) ctrlParms.string, ctrlParms.strLen );
            ctrlParms.result = 0;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_SET_MEMORY:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            unsigned long  *pul = (unsigned long *)  ctrlParms.string;
            unsigned short *pus = (unsigned short *) ctrlParms.string;
            unsigned char  *puc = (unsigned char *)  ctrlParms.string;
            switch( ctrlParms.strLen ) {
            case 4:
                *pul = (unsigned long) ctrlParms.offset;
                break;
            case 2:
                *pus = (unsigned short) ctrlParms.offset;
                break;
            case 1:
                *puc = (unsigned char) ctrlParms.offset;
                break;
            }
#if !defined(CONFIG_BCM96816) && !defined(CONFIG_BCM96818)
            /* This is placed as MoCA blocks are 32-bit only
            * accessible and following call makes access in terms
            * of bytes. Probably MoCA address range can be checked
            * here.
            */
            dumpaddr( (unsigned char *) ctrlParms.string, sizeof(long) );
#endif
            ctrlParms.result = 0;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_MIPS_SOFT_RESET:
        kerSysMipsSoftReset();
        break;

    case BOARD_IOCTL_LED_CTRL:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            kerSysLedCtrl((BOARD_LED_NAME)ctrlParms.strLen, (BOARD_LED_STATE)ctrlParms.offset);
            ret = 0;
        }
        break;

    case BOARD_IOCTL_GET_ID:
        if (copy_from_user((void*)&ctrlParms, (void*)arg,
            sizeof(ctrlParms)) == 0)
        {
            if( ctrlParms.string )
            {
                char p[NVRAM_BOARD_ID_STRING_LEN];
                kerSysNvRamGetBoardId(p);
                if( strlen(p) + 1 < ctrlParms.strLen )
                    ctrlParms.strLen = strlen(p) + 1;
                __copy_to_user(ctrlParms.string, p, ctrlParms.strLen);
            }

            ctrlParms.result = 0;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,
                sizeof(BOARD_IOCTL_PARMS));
        }
        break;

    case BOARD_IOCTL_GET_MAC_ADDRESS:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            ctrlParms.result = kerSysGetMacAddress( ucaMacAddr,
                ctrlParms.offset );

            if( ctrlParms.result == 0 )
            {
                __copy_to_user(ctrlParms.string, ucaMacAddr,
                    sizeof(ucaMacAddr));
            }

            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,
                sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_ALLOC_MAC_ADDRESSES:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            ctrlParms.result = kerSysGetMacAddresses( ucaMacAddr,
                *((UINT32 *)ctrlParms.buf), ctrlParms.offset );

            if( ctrlParms.result == 0 )
            {
                __copy_to_user(ctrlParms.string, ucaMacAddr,
                    sizeof(ucaMacAddr));
                ret = 0;
            } else {
                ret = -EFAULT;
            }

            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,
                sizeof(BOARD_IOCTL_PARMS));
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_RELEASE_MAC_ADDRESSES:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            if (copy_from_user((void*)ucaMacAddr, (void*)ctrlParms.string, \
                NVRAM_MAC_ADDRESS_LEN) == 0)
            {
                ctrlParms.result = kerSysReleaseMacAddresses( ucaMacAddr, *((UINT32 *)ctrlParms.buf) );
            }
            else
            {
                ctrlParms.result = -EACCES;
            }

            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,
                sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_RELEASE_MAC_ADDRESS:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            if (copy_from_user((void*)ucaMacAddr, (void*)ctrlParms.string, \
                NVRAM_MAC_ADDRESS_LEN) == 0)
            {
                ctrlParms.result = kerSysReleaseMacAddress( ucaMacAddr );
            }
            else
            {
                ctrlParms.result = -EACCES;
            }

            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,
                sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_GET_PSI_SIZE:
        {
            FLASH_ADDR_INFO fInfo;
            kerSysFlashAddrInfoGet(&fInfo);
            ctrlParms.result = fInfo.flash_persistent_length;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        break;

#ifdef ATI_PRODUCT_CONFIG
    case BOARD_IOCTL_GET_PSP_SIZE:
        {
            FLASH_ADDR_INFO fInfo;
            kerSysFlashAddrInfoGet(&fInfo);
            ctrlParms.result = fInfo.flash_scratch_pad_length;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        break;
#endif // ATI_PRODUCT_CONFIG

    case BOARD_IOCTL_GET_BACKUP_PSI_SIZE:
        {
            FLASH_ADDR_INFO fInfo;
            kerSysFlashAddrInfoGet(&fInfo);
            // if number_blks > 0, that means there is a backup psi, but length is the same
            // as the primary psi (persistent).

            ctrlParms.result = (fInfo.flash_backup_psi_number_blk > 0) ?
                fInfo.flash_persistent_length : 0;
            printk("backup_psi_number_blk=%d result=%d\n", fInfo.flash_backup_psi_number_blk, fInfo.flash_persistent_length);
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        break;

    case BOARD_IOCTL_GET_SYSLOG_SIZE:
        {
            FLASH_ADDR_INFO fInfo;
            kerSysFlashAddrInfoGet(&fInfo);
            ctrlParms.result = fInfo.flash_syslog_length;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        break;

    case BOARD_IOCTL_GET_SDRAM_SIZE:
        ctrlParms.result = (int) g_ulSdramSize;
        __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        ret = 0;
        break;

    case BOARD_IOCTL_GET_BASE_MAC_ADDRESS:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            __copy_to_user(ctrlParms.string, g_pMacInfo->ucaBaseMacAddr, NVRAM_MAC_ADDRESS_LEN);
            ctrlParms.result = 0;

            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,
                sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else
            ret = -EFAULT;
        break;

    case BOARD_IOCTL_GET_CHIP_ID:
        ctrlParms.result = kerSysGetChipId();


        __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        ret = 0;
        break;

    case BOARD_IOCTL_GET_CHIP_REV:
        ctrlParms.result = (int) (PERF->RevID & REV_ID_MASK);
        __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        ret = 0;
        break;

    case BOARD_IOCTL_GET_NUM_ENET_MACS:
    case BOARD_IOCTL_GET_NUM_ENET_PORTS:
        {
            ETHERNET_MAC_INFO EnetInfos[BP_MAX_ENET_MACS];
            int i, cnt, numEthPorts = 0;
            if (BpGetEthernetMacInfo(EnetInfos, BP_MAX_ENET_MACS) == BP_SUCCESS) {
                for( i = 0; i < BP_MAX_ENET_MACS; i++) {
                    if (EnetInfos[i].ucPhyType != BP_ENET_NO_PHY) {
                        bitcount(cnt, EnetInfos[i].sw.port_map);
                        numEthPorts += cnt;
                    }
                }
                ctrlParms.result = numEthPorts;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,  sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            }
            else {
                ret = -EFAULT;
            }
            break;
        }

    case BOARD_IOCTL_GET_CFE_VER:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            char vertag[CFE_VERSION_MARK_SIZE+CFE_VERSION_SIZE];
            kerSysCfeVersionGet(vertag, sizeof(vertag));
            if (ctrlParms.strLen < CFE_VERSION_SIZE) {
                ctrlParms.result = 0;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
                ret = -EFAULT;
            }
            else if (strncmp(vertag, "cfe-v", 5)) { // no tag info in flash
                ctrlParms.result = 0;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            }
            else {
                ctrlParms.result = 1;
                __copy_to_user(ctrlParms.string, vertag+CFE_VERSION_MARK_SIZE, CFE_VERSION_SIZE);
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            }
        }
        else {
            ret = -EFAULT;
        }
        break;

#if defined (WIRELESS)
    case BOARD_IOCTL_GET_WLAN_ANT_INUSE:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            unsigned short antInUse = 0;
            if (BpGetWirelessAntInUse(&antInUse) == BP_SUCCESS) {
                if (ctrlParms.strLen == sizeof(antInUse)) {
                    __copy_to_user(ctrlParms.string, &antInUse, sizeof(antInUse));
                    ctrlParms.result = 0;
                    __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
                    ret = 0;
                } else
                    ret = -EFAULT;
            }
            else {
                ret = -EFAULT;
            }
            break;
        }
        else {
            ret = -EFAULT;
        }
        break;
#endif
    case BOARD_IOCTL_SET_TRIGGER_EVENT:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            BOARD_IOC *board_ioc = (BOARD_IOC *)flip->private_data;
            ctrlParms.result = -EFAULT;
            ret = -EFAULT;
            if (ctrlParms.strLen == sizeof(unsigned long)) {
                board_ioc->eventmask |= *((int*)ctrlParms.string);
#if defined (WIRELESS)
                if((board_ioc->eventmask & SES_EVENTS)) {
                    if(sesBtn_irq != BP_NOT_DEFINED) {
                        BcmHalInterruptEnable(sesBtn_irq);
                        ctrlParms.result = 0;
                        ret = 0;
                    }
                }
#endif
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            }
            break;
        }
        else {
            ret = -EFAULT;
        }
        break;

    case BOARD_IOCTL_GET_TRIGGER_EVENT:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            BOARD_IOC *board_ioc = (BOARD_IOC *)flip->private_data;
            if (ctrlParms.strLen == sizeof(unsigned long)) {
                __copy_to_user(ctrlParms.string, &board_ioc->eventmask, sizeof(unsigned long));
                ctrlParms.result = 0;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            } else
                ret = -EFAULT;

            break;
        }
        else {
            ret = -EFAULT;
        }
        break;

    case BOARD_IOCTL_UNSET_TRIGGER_EVENT:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            if (ctrlParms.strLen == sizeof(unsigned long)) {
                BOARD_IOC *board_ioc = (BOARD_IOC *)flip->private_data;
                board_ioc->eventmask &= (~(*((int*)ctrlParms.string)));
                ctrlParms.result = 0;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            } else
                ret = -EFAULT;

            break;
        }
        else {
            ret = -EFAULT;
        }
        break;
#if defined (WIRELESS)
    case BOARD_IOCTL_SET_SES_LED:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            if (ctrlParms.strLen == sizeof(int)) {
                sesLed_ctrl(*(int*)ctrlParms.string);
                ctrlParms.result = 0;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            } else
                ret = -EFAULT;

            break;
        }
        else {
            ret = -EFAULT;
        }
        break;
#endif

    case BOARD_IOCTL_SET_MONITOR_FD:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {

           g_monitor_nl_pid =  ctrlParms.offset;
           printk(KERN_INFO "monitor task is initialized pid= %d \n",g_monitor_nl_pid);
        }
        break;

    case BOARD_IOCTL_WAKEUP_MONITOR_TASK:
        kerSysSendtoMonitorTask(MSG_NETLINK_BRCM_WAKEUP_MONITOR_TASK, NULL, 0);
        break;

    case BOARD_IOCTL_SET_CS_PAR:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            ret = ConfigCs(&ctrlParms);
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        }
        else {
            ret = -EFAULT;
        }
        break;

    case BOARD_IOCTL_SET_GPIO:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            kerSysSetGpioState(ctrlParms.strLen, ctrlParms.offset);
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else {
            ret = -EFAULT;
        }
        break;

#if defined(CONFIG_BCM_CPLD1)
    case BOARD_IOCTL_SET_SHUTDOWN_MODE:
        BcmCpld1SetShutdownMode();
        ret = 0;
        break;

    case BOARD_IOCTL_SET_STANDBY_TIMER:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            BcmCpld1SetStandbyTimer(ctrlParms.offset);
            ret = 0;
        }
        else {
            ret = -EFAULT;
        }
        break;
#endif

    case BOARD_IOCTL_BOOT_IMAGE_OPERATION:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            switch(ctrlParms.offset)
            {
            case BOOT_SET_PART1_IMAGE:
            case BOOT_SET_PART2_IMAGE:
            case BOOT_SET_PART1_IMAGE_ONCE:
            case BOOT_SET_PART2_IMAGE_ONCE:
            case BOOT_SET_OLD_IMAGE:
            case BOOT_SET_NEW_IMAGE:
            case BOOT_SET_NEW_IMAGE_ONCE:
                ctrlParms.result = kerSysSetBootImageState(ctrlParms.offset);
                break;

            case BOOT_GET_BOOT_IMAGE_STATE:
                ctrlParms.result = kerSysGetBootImageState();
                break;

            case BOOT_GET_IMAGE_VERSION:
                /* ctrlParms.action is parition number */
                ctrlParms.result = getImageVersion((int) ctrlParms.action,
                    ctrlParms.string, ctrlParms.strLen);
                break;

            case BOOT_GET_BOOTED_IMAGE_ID:
                /* ctrlParm.strLen == 1: partition or == 0: id (new or old) */
                ctrlParms.result = getBootedValue(ctrlParms.strLen);
                break;

            default:
                ctrlParms.result = -EFAULT;
                break;
            }
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else {
            ret = -EFAULT;
        }
        break;

    case BOARD_IOCTL_GET_TIMEMS:
        ret = jiffies_to_msecs(jiffies - INITIAL_JIFFIES);
        break;

    case BOARD_IOCTL_GET_DEFAULT_OPTICAL_PARAMS:
    {
        unsigned char ucDefaultOpticalParams[BP_OPTICAL_PARAMS_LEN];
            
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
            ret = 0;
            if (BP_SUCCESS == (ctrlParms.result = BpGetDefaultOpticalParams(ucDefaultOpticalParams)))
            {
                __copy_to_user(ctrlParms.string, ucDefaultOpticalParams, BP_OPTICAL_PARAMS_LEN);

                if (__copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS)) != 0)
                {
                    ret = -EFAULT;
                }
            }                        
        }
        else
        {
            ret = -EFAULT;
        }

        break;
    }
    
    break;
    case BOARD_IOCTL_GET_GPON_OPTICS_TYPE:
     
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) 
        {
            unsigned short Temp=0;
            BpGetGponOpticsType(&Temp);
            *((UINT32*)ctrlParms.buf) = Temp;
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
        }        
        ret = 0;

        break;

#if defined(CONFIG_BCM96816) || defined(CONFIG_BCM963268)
    case BOARD_IOCTL_SPI_SLAVE_INIT:  
        ret = 0;
        if (kerSysBcmSpiSlaveInit() != SPI_STATUS_OK)  
        {
            ret = -EFAULT;
        }        
        break;   
        
    case BOARD_IOCTL_SPI_SLAVE_READ:  
        ret = 0;
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
             if (kerSysBcmSpiSlaveRead(ctrlParms.offset, (unsigned long *)ctrlParms.buf, ctrlParms.strLen) != SPI_STATUS_OK)  
             {
                 ret = -EFAULT;
             } 
             else
             {
                   __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));    
             }
        }
        else
        {
            ret = -EFAULT;
        }                 
        break;    
        
    case BOARD_IOCTL_SPI_SLAVE_WRITE:  
        ret = 0;
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0)
        {
             if (kerSysBcmSpiSlaveWrite(ctrlParms.offset, ctrlParms.result, ctrlParms.strLen) != SPI_STATUS_OK)  
             {
                 ret = -EFAULT;
             } 
             else
             {
                   __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));    
             }
        }
        else
        {
            ret = -EFAULT;
        }                 
        break;    
#endif

#if defined(CONFIG_EPON_SDK)
    case BOARD_IOCTL_GET_NUM_FE_PORTS:
        {
            unsigned long fe_ports;
            if (BpGetNumFePorts(&fe_ports) == BP_SUCCESS) {
                ctrlParms.result = fe_ports;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,  sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            }
            else {
                ret = -EFAULT;
            }
            break;
        }

    case BOARD_IOCTL_GET_NUM_GE_PORTS:
        {
            unsigned long ge_ports;
            if (BpGetNumGePorts(&ge_ports) == BP_SUCCESS) {
                ctrlParms.result = ge_ports;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,  sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            }
            else {
                ret = -EFAULT;
            }
            break;
        }

    case BOARD_IOCTL_GET_NUM_VOIP_PORTS:
        {
            unsigned long voip_ports;
            if (BpGetNumVoipPorts(&voip_ports) == BP_SUCCESS) {
                ctrlParms.result = voip_ports;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,  sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            }
            else {
                ret = -EFAULT;
            }
            break;
        }

    case BOARD_IOCTL_GET_SWITCH_PORT_MAP:
        {
            unsigned long port_map;
            if (BpGetSwitchPortMap(&port_map) == BP_SUCCESS) {
                ctrlParms.result = port_map;
                __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms,  sizeof(BOARD_IOCTL_PARMS));
                ret = 0;
            }
            else {
                ret = -EFAULT;
            }
            break;
        }
#endif

    case BOARD_IOCTL_SET_WATCHDOG:
        {
            if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) 
            {
                switch(ctrlParms.action)
                {
                case WD_INIT:
                    TIMER->WatchDogDefCount = (uint32)ctrlParms.buf;
                    TIMER->WatchDogCtl = 0xff00;
                    TIMER->WatchDogCtl = 0x00ff;
                    break;
                case WD_TICKLE:
                    TIMER->WatchDogCtl = 0xff00;
                    TIMER->WatchDogCtl = 0x00ff;
                    break;
                default:
                    break;
                }
            }        
            ret = 0;
            break;
        }
    case BOARD_IOCTL_GET_GPIO:
        if (copy_from_user((void*)&ctrlParms, (void*)arg, sizeof(ctrlParms)) == 0) {
            ctrlParms.result = kerSysGetGpioState(ctrlParms.offset);
            __copy_to_user((BOARD_IOCTL_PARMS*)arg, &ctrlParms, sizeof(BOARD_IOCTL_PARMS));
            ret = 0;
        }
        else {
            ret = -EFAULT;
        }
        break;
    default:
        ret = -EINVAL;
        ctrlParms.result = 0;
        printk("board_ioctl: invalid command %x, cmd %d .\n",command,_IOC_NR(command));
        break;

    } /* switch */

    return (ret);

} /* board_ioctl */

/***************************************************************************
* SES Button ISR/GPIO/LED functions.
***************************************************************************/
#if defined (WIRELESS)

static Bool sesBtn_pressed(void)
{
    if ((sesBtn_irq >= INTERRUPT_ID_EXTERNAL_0) && (sesBtn_irq <= INTERRUPT_ID_EXTERNAL_3)) {
        if (!(PERF->ExtIrqCfg & (1 << (sesBtn_irq - INTERRUPT_ID_EXTERNAL_0 + EI_STATUS_SHFT)))) {
            return 1;
        }
    }
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816) || defined(CONFIG_BCM96818)
    else if ((sesBtn_irq >= INTERRUPT_ID_EXTERNAL_4) || (sesBtn_irq <= INTERRUPT_ID_EXTERNAL_5)) {
        if (!(PERF->ExtIrqCfg1 & (1 << (sesBtn_irq - INTERRUPT_ID_EXTERNAL_4 + EI_STATUS_SHFT)))) {
            return 1;
        }
    }
#endif
    return 0;
}

static irqreturn_t sesBtn_isr(int irq, void *dev_id)
{
    if (sesBtn_pressed()){
        wake_up_interruptible(&g_board_wait_queue);
        return IRQ_RETVAL(1);
    } else {
        return IRQ_RETVAL(0);
    }
}

static void __init sesBtn_mapIntr(int context)
{
    if( BpGetWirelessSesExtIntr(&sesBtn_irq) == BP_SUCCESS )
    {
        printk("SES: Button Interrupt 0x%x is enabled\n", sesBtn_irq);
    }
    else
        return;

    sesBtn_irq = map_external_irq (sesBtn_irq) ;

    if (BcmHalMapInterrupt((FN_HANDLER)sesBtn_isr, context, sesBtn_irq)) {
        printk("SES: Interrupt mapping failed\n");
    }
    BcmHalInterruptEnable(sesBtn_irq);
}


static unsigned int sesBtn_poll(struct file *file, struct poll_table_struct *wait)
{

    if (sesBtn_pressed()){
        return POLLIN;
    }
    return 0;
}

static ssize_t sesBtn_read(struct file *file,  char __user *buffer, size_t count, loff_t *ppos)
{
    volatile unsigned int event=0;
    ssize_t ret=0;

    if(!sesBtn_pressed()){
        BcmHalInterruptEnable(sesBtn_irq);
        return ret;
    }
    event = SES_EVENTS;
    __copy_to_user((char*)buffer, (char*)&event, sizeof(event));
    BcmHalInterruptEnable(sesBtn_irq);
    count -= sizeof(event);
    buffer += sizeof(event);
    ret += sizeof(event);
    return ret;
}

static void __init sesLed_mapGpio()
{
    if( BpGetWirelessSesLedGpio(&sesLed_gpio) == BP_SUCCESS )
    {
        printk("SES: LED GPIO 0x%x is enabled\n", sesLed_gpio);
    }
}

static void sesLed_ctrl(int action)
{
    char blinktype = ((action >> 24) & 0xff); /* extract blink type for SES_LED_BLINK  */

    BOARD_LED_STATE led;

    if(sesLed_gpio == BP_NOT_DEFINED)
        return;

    action &= 0xff; /* extract led */

    switch (action) {
    case SES_LED_ON:
        led = kLedStateOn;
        break;
    case SES_LED_BLINK:
        if(blinktype)
            led = blinktype;
        else
            led = kLedStateSlowBlinkContinues;           		
        break;
    case SES_LED_OFF:
    default:
        led = kLedStateOff;
    }

    kerSysLedCtrl(kLedSes, led);
}

static void __init ses_board_init()
{
    sesBtn_mapIntr(0);
    sesLed_mapGpio();
}
static void __exit ses_board_deinit()
{
    if(sesBtn_irq)
        BcmHalInterruptDisable(sesBtn_irq);
}
#endif

/***************************************************************************
* Dying gasp ISR and functions.
***************************************************************************/

static irqreturn_t kerSysDyingGaspIsr(int irq, void * dev_id)
{
    struct list_head *pos;
    CB_DGASP_LIST *tmp = NULL, *dslOrGpon = NULL;
	unsigned short usPassDyingGaspGpio;		// The GPIO pin to propogate a dying gasp signal

    UART->Data = 'D';
    UART->Data = '%';
    UART->Data = 'G';

#if defined (WIRELESS)
    kerSetWirelessPD(WLAN_OFF);
#endif
    /* first to turn off everything other than dsl or gpon */
    list_for_each(pos, &g_cb_dgasp_list_head->list) {
        tmp = list_entry(pos, CB_DGASP_LIST, list);
        if(strncmp(tmp->name, "dsl", 3) && strncmp(tmp->name, "gpon", 4)) {
            (tmp->cb_dgasp_fn)(tmp->context);
        }else {
            dslOrGpon = tmp;
        }
    }
	
    // Invoke dying gasp handlers
    if(dslOrGpon)
        (dslOrGpon->cb_dgasp_fn)(dslOrGpon->context);

    /* reset and shutdown system */
    /* Set WD to fire in 1 sec in case power is restored before reset occurs */
    TIMER->WatchDogDefCount = 1000000 * (FPERIPH/1000000);
    TIMER->WatchDogCtl = 0xFF00;
    TIMER->WatchDogCtl = 0x00FF;

	// If configured, propogate dying gasp to other processors on the board
	if(BpGetPassDyingGaspGpio(&usPassDyingGaspGpio) == BP_SUCCESS)
	    {
	    // Dying gasp configured - set GPIO
	    kerSysSetGpioState(usPassDyingGaspGpio, kGpioInactive);
	    }

    // If power is going down, nothing should continue!
    while (1);
    return( IRQ_HANDLED );
}

static void __init kerSysInitDyingGaspHandler( void )
{
    CB_DGASP_LIST *new_node;

    return;

    if( g_cb_dgasp_list_head != NULL) {
        printk("Error: kerSysInitDyingGaspHandler: list head is not null\n");
        return;
    }
    new_node= (CB_DGASP_LIST *)kmalloc(sizeof(CB_DGASP_LIST), GFP_KERNEL);
    memset(new_node, 0x00, sizeof(CB_DGASP_LIST));
    INIT_LIST_HEAD(&new_node->list);
    g_cb_dgasp_list_head = new_node;

    BcmHalMapInterrupt((FN_HANDLER)kerSysDyingGaspIsr, 0, INTERRUPT_ID_DG);
    BcmHalInterruptEnable( INTERRUPT_ID_DG );
} /* kerSysInitDyingGaspHandler */

static void __exit kerSysDeinitDyingGaspHandler( void )
{
    struct list_head *pos;
    CB_DGASP_LIST *tmp;

    if(g_cb_dgasp_list_head == NULL)
        return;

    list_for_each(pos, &g_cb_dgasp_list_head->list) {
        tmp = list_entry(pos, CB_DGASP_LIST, list);
        list_del(pos);
        kfree(tmp);
    }

    kfree(g_cb_dgasp_list_head);
    g_cb_dgasp_list_head = NULL;

} /* kerSysDeinitDyingGaspHandler */

void kerSysRegisterDyingGaspHandler(char *devname, void *cbfn, void *context)
{
    CB_DGASP_LIST *new_node;

    // do all the stuff that can be done without the lock first
    if( devname == NULL || cbfn == NULL ) {
        printk("Error: kerSysRegisterDyingGaspHandler: register info not enough (%s,%x,%x)\n", devname, (unsigned int)cbfn, (unsigned int)context);
        return;
    }

    if (strlen(devname) > (IFNAMSIZ - 1)) {
        printk("Warning: kerSysRegisterDyingGaspHandler: devname too long, will be truncated\n");
    }

    new_node= (CB_DGASP_LIST *)kmalloc(sizeof(CB_DGASP_LIST), GFP_KERNEL);
    memset(new_node, 0x00, sizeof(CB_DGASP_LIST));
    INIT_LIST_HEAD(&new_node->list);
    strncpy(new_node->name, devname, IFNAMSIZ-1);
    new_node->cb_dgasp_fn = (cb_dgasp_t)cbfn;
    new_node->context = context;

    // OK, now acquire the lock and insert into list
    mutex_lock(&dgaspMutex);
    if( g_cb_dgasp_list_head == NULL) {
        printk("Error: kerSysRegisterDyingGaspHandler: list head is null\n");
        kfree(new_node);
    } else {
        list_add(&new_node->list, &g_cb_dgasp_list_head->list);
        printk("dgasp: kerSysRegisterDyingGaspHandler: %s registered \n", devname);
    }
    mutex_unlock(&dgaspMutex);

    return;
} /* kerSysRegisterDyingGaspHandler */

void kerSysDeregisterDyingGaspHandler(char *devname)
{
    struct list_head *pos;
    CB_DGASP_LIST *tmp;
    int found=0;

    if(devname == NULL) {
        printk("Error: kerSysDeregisterDyingGaspHandler: devname is null\n");
        return;
    }

    printk("kerSysDeregisterDyingGaspHandler: %s is deregistering\n", devname);

    mutex_lock(&dgaspMutex);
    if(g_cb_dgasp_list_head == NULL) {
        printk("Error: kerSysDeregisterDyingGaspHandler: list head is null\n");
    } else {
        list_for_each(pos, &g_cb_dgasp_list_head->list) {
            tmp = list_entry(pos, CB_DGASP_LIST, list);
            if(!strcmp(tmp->name, devname)) {
                list_del(pos);
                kfree(tmp);
                found = 1;
                printk("kerSysDeregisterDyingGaspHandler: %s is deregistered\n", devname);
                break;
            }
        }
        if (!found)
            printk("kerSysDeregisterDyingGaspHandler: %s not (de)registered\n", devname);
    }
    mutex_unlock(&dgaspMutex);

    return;
} /* kerSysDeregisterDyingGaspHandler */


/***************************************************************************
 *
 *
 ***************************************************************************/
static int ConfigCs (BOARD_IOCTL_PARMS *parms)
{
    int                     retv = 0;
#if defined(CONFIG_BCM96368) || defined(CONFIG_BCM96816) || defined(CONFIG_BCM96818)
    int                     cs, flags;
    cs_config_pars_t        info;

    if (copy_from_user(&info, (void*)parms->buf, sizeof(cs_config_pars_t)) == 0)
    {
        cs = parms->offset;

        MPI->cs[cs].base = ((info.base & 0x1FFFE000) | (info.size >> 13));

        if ( info.mode == EBI_TS_TA_MODE )     // syncronious mode
            flags = (EBI_TS_TA_MODE | EBI_ENABLE);
        else
        {
            flags = ( EBI_ENABLE | \
                (EBI_WAIT_STATES  & (info.wait_state << EBI_WTST_SHIFT )) | \
                (EBI_SETUP_STATES & (info.setup_time << EBI_SETUP_SHIFT)) | \
                (EBI_HOLD_STATES  & (info.hold_time  << EBI_HOLD_SHIFT )) );
        }
        MPI->cs[cs].config = flags;
        parms->result = BP_SUCCESS;
        retv = 0;
    }
    else
    {
        retv -= EFAULT;
        parms->result = BP_NOT_DEFINED;
    }
#endif
    return( retv );
}


/***************************************************************************
* Handle push of restore to default button
***************************************************************************/
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 20)
static void restore_to_default_thread(struct work_struct *work)
#else
static void restore_to_default_thread(void *arg)
#endif
{
    char buf[256];

    memset(buf, 0, sizeof(buf));

#if !defined(ATI_BSP_PERSONALITY)
// The restore defaults is handled via UBOOT and is a timed event
    // Do this in a kernel thread so we don't have any restriction
    printk("Restore to Factory Default Setting ***\n\n");
    kerSysPersistentSet( buf, sizeof(buf), 0 );
#endif
    // kernel_restart is a high level, generic linux way of rebooting.
    // It calls a notifier list and lets sub-systems know that system is
    // rebooting, and then calls machine_restart, which eventually
    // calls kerSysMipsSoftReset.
    kernel_restart("reason=button");
    return;
}

static irqreturn_t reset_isr(int irq, void *dev_id)
{
    printk("\n***reset button press detected***\n\n");

    INIT_WORK(&restoreDefaultWork, restore_to_default_thread);
    schedule_work(&restoreDefaultWork);
    return IRQ_HANDLED;
}



#if defined(WIRELESS)
/***********************************************************************
* Function Name: kerSysScreenPciDevices
* Description  : Screen Pci Devices before loading modules
***********************************************************************/
static void __init kerSysScreenPciDevices(void)
{
    unsigned short wlFlag;

    if((BpGetWirelessFlags(&wlFlag) == BP_SUCCESS) && (wlFlag & BP_WLAN_EXCLUDE_ONBOARD)) {
        /*
        * scan all available pci devices and delete on board BRCM wireless device
        * if external slot presents a BRCM wireless device
        */
        int foundPciAddOn = 0;
        struct pci_dev *pdevToExclude = NULL;
        struct pci_dev *dev = NULL;

        while((dev=pci_get_device(PCI_ANY_ID, PCI_ANY_ID, dev))!=NULL) {
            printk("kerSysScreenPciDevices: 0x%x:0x%x:(slot %d) detected\n", dev->vendor, dev->device, PCI_SLOT(dev->devfn));
            if((dev->vendor == BRCM_VENDOR_ID) &&
                (((dev->device & 0xff00) == BRCM_WLAN_DEVICE_IDS)|| 
                ((dev->device/1000) == BRCM_WLAN_DEVICE_IDS_DEC))) {
                    if(PCI_SLOT(dev->devfn) != WLAN_ONBOARD_SLOT) {
                        foundPciAddOn++;
                    } else {
                        pdevToExclude = dev;
                    }                
            }
        }

        if(((wlFlag & BP_WLAN_EXCLUDE_ONBOARD_FORCE) || foundPciAddOn) && pdevToExclude) {
            printk("kerSysScreenPciDevices: 0x%x:0x%x:(onboard) deleted\n", pdevToExclude->vendor, pdevToExclude->device);
            pci_remove_bus_device(pdevToExclude);
        }
    }
}

/***********************************************************************
* Function Name: kerSetWirelessPD
* Description  : Control Power Down by Hardware if the board supports
***********************************************************************/
static void kerSetWirelessPD(int state)
{
    unsigned short wlanPDGpio;
    if((BpGetWirelessPowerDownGpio(&wlanPDGpio)) == BP_SUCCESS) {
        if (wlanPDGpio != BP_NOT_DEFINED) {
            if(state == WLAN_OFF)
                kerSysSetGpioState(wlanPDGpio, kGpioActive);
            else
                kerSysSetGpioState(wlanPDGpio, kGpioInactive);
        }
    }
}

#endif


#if defined(PCIE_BASE)
/***********************************************************************
* Function Name: kerSysCheckPowerDownPcie
* Description  : Power Down PCIe if no device enumerated
*                Otherwise enable Power Saving modes
***********************************************************************/
static void __init kerSysCheckPowerDownPcie(void)
{
    struct pci_dev *dev = NULL;

    while ((dev=pci_get_device(PCI_ANY_ID, PCI_ANY_ID, dev))!=NULL) {
        if(BCM_BUS_PCIE_DEVICE == dev->bus->number) {
            /* Enable PCIe L1 PLL power savings */
            PCIEH_BLK_1800_REGS->phyCtrl[1] |= REG_POWERDOWN_P1PLL_ENA;
#if defined(PCIE_BRIDGE_CLKREQ_ENABLE)
            {
                unsigned long GPIOOverlays;

                /* Enable PCIe CLKREQ# power savings */
                if( (BpGetGPIOverlays(&GPIOOverlays) == BP_SUCCESS) && (GPIOOverlays & BP_OVERLAY_PCIE_CLKREQ)) {
                    PCIEH_BRIDGE_REGS->pcieControl |= PCIE_BRIDGE_CLKREQ_ENABLE;
                }
            }
#endif
            return;
        }
    }
            
    printk("PCIe: No device found - Powering down\n");
    /* pcie clock disable*/
    PERF->blkEnables &= ~PCIE_CLK_EN;
#if defined(CONFIG_BCM963268)
    MISC->miscLcpll_ctrl |= MISC_CLK100_DISABLE;
#endif

    /* pcie serdes disable */
#if defined(CONFIG_BCM96816) 
    GPIO->SerdesCtl &= ~(SERDES_PCIE_ENABLE|SERDES_PCIE_EXD_ENABLE);
#elif defined(CONFIG_BCM96818) 
    GPIO->SerdesCtl &= ~SERDES_PCIE_ENABLE;
#else
    MISC->miscSerdesCtrl &= ~(SERDES_PCIE_ENABLE|SERDES_PCIE_EXD_ENABLE);
#endif

    /* pcie and ext device */
    PERF->softResetB &= ~(SOFT_RST_PCIE|SOFT_RST_PCIE_EXT|SOFT_RST_PCIE_CORE);
#if defined(CONFIG_BCM96328) || defined(CONFIG_BCM96318)
    PERF->softResetB &= ~SOFT_RST_PCIE_HARD;
#endif

}
#endif

extern unsigned char g_blparms_buf[];

/***********************************************************************
 * Function Name: kerSysBlParmsGetInt
 * Description  : Returns the integer value for the requested name from
 *                the boot loader parameter buffer.
 * Returns      : 0 - success, -1 - failure
 ***********************************************************************/
int kerSysBlParmsGetInt( char *name, int *pvalue )
{
    char *p2, *p1 = g_blparms_buf;
    int ret = -1;

    *pvalue = -1;

    /* The g_blparms_buf buffer contains one or more contiguous NULL termianted
     * strings that ends with an empty string.
     */
    while( *p1 )
    {
        p2 = p1;

        while( *p2 != '=' && *p2 != '\0' )
            p2++;

        if( *p2 == '=' )
        {
            *p2 = '\0';

            if( !strcmp(p1, name) )
            {
                *p2++ = '=';
                *pvalue = simple_strtol(p2, &p1, 0);
                if( *p1 == '\0' )
                    ret = 0;
                break;
            }

            *p2 = '=';
        }

        p1 += strlen(p1) + 1;
    }

    return( ret );
}

/***********************************************************************
 * Function Name: kerSysBlParmsGetStr
 * Description  : Returns the string value for the requested name from
 *                the boot loader parameter buffer.
 * Returns      : 0 - success, -1 - failure
 ***********************************************************************/
int kerSysBlParmsGetStr( char *name, char *pvalue, int size )
{
    char *p2, *p1 = g_blparms_buf;
    int ret = -1;

    /* The g_blparms_buf buffer contains one or more contiguous NULL termianted
     * strings that ends with an empty string.
     */
    while( *p1 )
    {
        p2 = p1;

        while( *p2 != '=' && *p2 != '\0' )
            p2++;

        if( *p2 == '=' )
        {
            *p2 = '\0';

            if( !strcmp(p1, name) )
            {
                *p2++ = '=';
                strncpy(pvalue, p2, size);
                ret = 0;
                break;
            }

            *p2 = '=';
        }

        p1 += strlen(p1) + 1;
    }

    return( ret );
}

static int add_proc_files(void)
{
#define offset(type, elem) ((int)&((type *)0)->elem)

#if !defined(ATI_BSP_PERSONALITY)
    static int BaseMacAddr[2] = {offset(NVRAM_DATA, ucaBaseMacAddr), NVRAM_MAC_ADDRESS_LEN};
#else
  static int BaseMacAddr[3] = {bcmEmac, offset(NVRAM_DATA, ucaBaseMacAddr), NVRAM_MAC_ADDRESS_LEN};
#endif

    struct proc_dir_entry *p0;
    struct proc_dir_entry *p1;

    p0 = proc_mkdir("nvram", NULL);

    if (p0 == NULL)
    {
        printk("add_proc_files: failed to create proc files!\n");
        return -1;
    }

    p1 = create_proc_entry("BaseMacAddr", 0644, p0);

    if (p1 == NULL)
    {
        printk("add_proc_files: failed to create proc files!\n");
        return -1;
    }

    p1->data        = BaseMacAddr;
    p1->read_proc   = proc_get_param;
    p1->write_proc  = proc_set_param;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
	//New linux no longer requires proc_dir_entry->owner field.
#else
    p1->owner       = THIS_MODULE;
#endif

    p1 = create_proc_entry("led", 0644, NULL);
    if (p1 == NULL)
        return -1;

#if defined(ATI_BSP_PERSONALITY)
    p1->data        = NULL;
    p1->read_proc   = proc_get_led;
#endif
    p1->write_proc  = proc_set_led;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
	//New linux no longer requires proc_dir_entry->owner field.
#else
    p1->owner       = THIS_MODULE;
#endif

#if defined(ATI_BSP_PERSONALITY)
  {
      int envCount = 0;
	  struct proc_dir_entry *proc_gpio_dir;
	  struct proc_dir_entry *proc_gpio_fentry;
      struct proc_dir_entry *proc_gpio_direction_fentry;
      char gpio_dir_fname[64+3];

	  // create /proc/gpio/ directory
	  proc_gpio_dir = proc_mkdir("gpio", NULL);
	  if (proc_gpio_dir== NULL) 
	  {
		  printk("add_proc_files: failed to create /proc/gpio/ directory!\n");
		  return -1;
	  }

      while (!(NvRamLookUpTable[envCount][0] == 0 && NvRamLookUpTable[envCount][1] == 0 && NvRamLookUpTable[envCount][2] == 0))
      {
          // Figure out whether a /proc/gpio/ entry needs to be created.
          // All GPIO environment vars start with "AtiGpio".
          if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiGpio", sizeof("AtiGpio")-1) == 0) {
              // Sanitize AtiGpio value. If value is invalid, no /proc/gpio/ entry is created.
              if (strlen(   &atiNvramData.buffer[NvRamLookUpTable[envCount][2]]) == 2 &&
                  isxdigit((&atiNvramData.buffer[NvRamLookUpTable[envCount][2]])[0]) &&
                  isxdigit((&atiNvramData.buffer[NvRamLookUpTable[envCount][2]])[1])) 
              {
                  proc_gpio_fentry = create_proc_entry(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], 0644, proc_gpio_dir);
                  if (proc_gpio_fentry == NULL)
                  {
                      printk("add_proc_files: failed to create /proc/gpio/%s file!\n", &atiNvramData.buffer[NvRamLookUpTable[envCount][1]]);
                  }
                  
                  // Also create a file to display/control GPIO direction
                  snprintf(gpio_dir_fname, sizeof(gpio_dir_fname), "%sDir", &atiNvramData.buffer[NvRamLookUpTable[envCount][1]]);
                  proc_gpio_direction_fentry = create_proc_entry(gpio_dir_fname, 0644, proc_gpio_dir);
                  if (proc_gpio_direction_fentry == NULL)
                  {
                      printk("add_proc_files: failed to create /proc/gpio/%s file!\n", gpio_dir_fname);
                  }

                  // The value of the AtiGpio environment variable is be passed to the proc_get_gpio call back function 
                  // after converting it from a string to an integer value. Casting long to int is OK. Max value is 256.
                  proc_gpio_fentry->data 		= (void *)simple_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], NULL, 16);
                  proc_gpio_fentry->read_proc 	= proc_get_gpio;
                  proc_gpio_fentry->write_proc	= proc_set_gpio;

                  proc_gpio_direction_fentry->data          = (void *)simple_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], NULL, 16);
                  proc_gpio_direction_fentry->read_proc 	= proc_get_gpio_dir;
                  proc_gpio_direction_fentry->write_proc	= proc_set_gpio_dir;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
              //New linux no longer requires proc_dir_entry->owner field.
#else
                  proc_gpio_fentry->owner       = THIS_MODULE;
#endif
              }
              else
              {
                  printk("add_proc_files: Invalid value (\"%s\") for environment variable %s!\n",&atiNvramData.buffer[NvRamLookUpTable[envCount][2]],&atiNvramData.buffer[NvRamLookUpTable[envCount][1]]);
              }
		  }
		  //printk("proc entry=%s (%s)\n", &atiNvramData.buffer[NvRamLookUpTable[envCount][1]],&atiNvramData.buffer[NvRamLookUpTable[envCount][2]]);
          p1 = create_proc_entry(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], 0644, p0);
          p1->data        = &NvRamLookUpTable[envCount][0];
          p1->read_proc   = proc_get_param;
          p1->write_proc  = proc_set_param;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
   	      //New linux no longer requires proc_dir_entry->owner field.
#else
          p1->owner       = THIS_MODULE;
#endif
          if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiResetReason", sizeof("AtiResetReason")) == 0) {
              int coldStart;
              coldStart = !strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], "coldstart", 9);
              if (coldStart)
              {
                  p1 = create_proc_entry("AtiColdStart", 0644, p0);
              }
              else
              {
                  p1 = create_proc_entry("AtiWarmStart", 0644, p0);
              }

              if (p1 == NULL)
              {
                printk("add_proc_files: failed to create proc files!\n");
                return -1;
              }

              p1->data        = &NvRamLookUpTable[envCount][0];
              p1->read_proc   = proc_get_param;
              p1->write_proc  = proc_set_param;

        #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
              //New linux no longer requires proc_dir_entry->owner field.
        #else
              p1->owner       = THIS_MODULE;
        #endif
          } else if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiKernelPersistentLogAddr", sizeof("AtiKernelPersistentLogAddr")) == 0) {
              unsigned mult;
              atiPersistentLogAddr = (unsigned *)simple_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], NULL, 10);
              //Find Multiplier
              mult = 1;
              if (strchr(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 'K'))
                  mult = 1024;
              if (strchr(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 'M'))
                  mult = 1024*1024;
              if (strchr(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 'G'))
                  mult = 1024*1024*1024;
              atiPersistentLogAddr = (unsigned *)((unsigned)atiPersistentLogAddr * mult + 0x80000000);
//              printk("atiPersistentLogAddr=%p\n",atiPersistentLogAddr);
              if (atiPersistentLogAddr && atiPersistentLogSize)
              {
                 atiPersLogMemMap =  (tAtiPersistentLogMemoryMap *) atiPersistentLogAddr;
              }
          } else if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiKernelPersistentLogSize", sizeof("AtiKernelPersistentLogSize")) == 0) {
              unsigned mult;
              atiPersistentLogSize = simple_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], NULL, 10);
              //Find Multiplier
              mult = 1;
              if (strchr(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 'K'))
                  mult = 1024;
              if (strchr(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 'M'))
                  mult = 1024*1024;
              if (strchr(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 'G'))
                  mult = 1024*1024*1024;
              atiPersistentLogSize = atiPersistentLogSize * mult;
//              printk("atiPersistentLogSize=%d\n",atiPersistentLogSize);
              if (atiPersistentLogAddr && atiPersistentLogSize)
              {
                 atiPersLogMemMap =  (tAtiPersistentLogMemoryMap *) atiPersistentLogAddr;
              }
          } else if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiKernelBootReasonAddr", sizeof("AtiKernelBootReasonAddr")) == 0) {
              atiKernelBootReasonAddr = (unsigned *)simple_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], NULL, 16);
//              printk("atiKernelBootReasonAddr=%p\n",atiKernelBootReasonAddr);
              if (atiKernelBootReasonAddr)
              {
                  atiResMemMap = (tAtiReservedMemoryMap *) atiKernelBootReasonAddr;
//                  printk("Init atiResMemMap %p\n", atiResMemMap);
              }
          }

//        We have decided not to uppercase teh MAC Address- since this could cause upgrade issues - due to IP Address changes
//        else if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiMacAddBase", sizeof("AtiMacAddBase")) == 0) {
//
//            char *pTmpChar = &atiNvramData.buffer[NvRamLookUpTable[envCount][2]];
//            // Change the mac address to lower case
//            for (; *pTmpChar; pTmpChar++)
//              *pTmpChar = tolower(*pTmpChar);
//          }

          if (p1 == NULL)
          {
            printk("add_proc_files: failed to create proc files!\n");
            return -1;
          }

          envCount++;
      }

#if defined(ATI_RELEASETAG)
      p1 = create_proc_entry("AtiReleaseTag", 0644, p0);

      if (p1 == NULL)
      {
        printk("add_proc_files: failed to create proc files!\n");
        return -1;
      }

      AtiTagData[0][0] = atiStringInd;
      AtiTagData[0][1] = strlen(ATI_RELEASETAG);
      AtiTagData[0][2] = (int) &ATI_RELEASETAG;

      p1->data        = &AtiTagData[0][0];
      p1->read_proc   = proc_get_param;
      p1->write_proc  = proc_set_param;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
      //New linux no longer requires proc_dir_entry->owner field.
#else
      p1->owner       = THIS_MODULE;
#endif
#endif
#if defined(ATI_VOIP_RELEASETAG)
      p1 = create_proc_entry("AtiVoipReleaseTag", 0644, p0);

      if (p1 == NULL)
      {
        printk("add_proc_files: failed to create proc files!\n");
        return -1;
      }

      AtiTagData[1][0] = atiStringInd;
      AtiTagData[1][1] = strlen(ATI_VOIP_RELEASETAG);
      AtiTagData[1][2] = (int)&ATI_VOIP_RELEASETAG;

      p1->data        = &AtiTagData[1][0];
      p1->read_proc   = proc_get_param;
      p1->write_proc  = proc_set_param;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
      //New linux no longer requires proc_dir_entry->owner field.
#else
      p1->owner       = THIS_MODULE;
#endif
#endif

#if defined(ATI_BUILD_TYPE)
      p1 = create_proc_entry("AtiReleaseType", 0644, p0);

      if (p1 == NULL)
      {
        printk("add_proc_files: failed to create proc files!\n");
        return -1;
      }

      AtiTagData[2][0] = atiStringInd;
      AtiTagData[2][1] = strlen(ATI_BUILD_TYPE);
      AtiTagData[2][2] = (int)&ATI_BUILD_TYPE;

      p1->data        = &AtiTagData[2][0];
      p1->read_proc   = proc_get_param;
      p1->write_proc  = proc_set_param;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
      //New linux no longer requires proc_dir_entry->owner field.
#else
      p1->owner       = THIS_MODULE;
#endif
#endif
 
#if defined(ATI_PHOENIX_RELEASETAG) 
      p1 = create_proc_entry("AtiCliReleaseTag", 0644, p0);

      if (p1 == NULL)
      {
        printk("add_proc_files: failed to create proc files!\n");
        return -1;
      }

      AtiTagData[3][0] = atiStringInd;
      AtiTagData[3][1] = strlen(ATI_PHOENIX_RELEASETAG);
      AtiTagData[3][2] = (int)&ATI_PHOENIX_RELEASETAG;

      p1->data        = &AtiTagData[3][0];
      p1->read_proc   = proc_get_param;
      p1->write_proc  = proc_set_param;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
      //New linux no longer requires proc_dir_entry->owner field.
#else
      p1->owner       = THIS_MODULE;
#endif
#endif

  }
#endif //#if ATI_BSP_PERSONALITY

    return 0;
}

static int del_proc_files(void)
{
    remove_proc_entry("nvram", NULL);
    remove_proc_entry("led", NULL);
	remove_proc_entry("gpio", NULL);
    return 0;
}

static void str_to_num(char* in, char* out, int len)
{
    int i;
    memset(out, 0, len);

    for (i = 0; i < len * 2; i ++)
    {
        if ((*in >= '0') && (*in <= '9'))
            *out += (*in - '0');
        else if ((*in >= 'a') && (*in <= 'f'))
            *out += (*in - 'a') + 10;
        else if ((*in >= 'A') && (*in <= 'F'))
            *out += (*in - 'A') + 10;
        else
            *out += 0;

        if ((i % 2) == 0)
            *out *= 16;
        else
            out ++;

        in ++;
    }
    return;
}

static int proc_get_param(char *page, char **start, off_t off, int cnt, int *eof, void *data)
{
#if !defined(ATI_BSP_PERSONALITY)
    int i = 0;
    int r = 0;
    int offset  = ((int *)data)[0];
    int length  = ((int *)data)[1];
    NVRAM_DATA *pNvramData;

    *eof = 1;

    if ((offset < 0) || (offset + length > sizeof(NVRAM_DATA)))
        return 0;

    if (NULL != (pNvramData = readNvramData()))
    {
        for (i = 0; i < length; i ++)
            r += sprintf(page + r, "%02x ", ((unsigned char *)pNvramData)[offset + i]);
    }

    r += sprintf(page + r, "\n");
    if (pNvramData)
        kfree(pNvramData);
    return (r < cnt)? r: 0;
#else
  int i = 0;
  int r = 0;
  ati_info_data_type atiType  = ((int *)data)[0];
  int offset  = ((int *)data)[2];

  switch (atiType)
  {
    case atiEmac:
    case atiStringInd:
      r += sprintf(page + r, "%s", (char *)offset);
      break;
    case atiString:
      r += sprintf(page + r, "%s", atiNvramData.buffer + offset);
      break;
    case atiUlong:
      r += sprintf(page + r, "%lx", *(unsigned long *)(atiNvramData.buffer + offset));
      break;
    case atiInt:
      r += sprintf(page + r, "%d", *(int *)(atiNvramData.buffer + offset));
      break;
    case bcmUlong:
    case bcmInt:
    case bcmString:
    case bcmTime:
    case bcmEmac:
    default:
    {
      int length  = ((int *)data)[2];
      offset  = ((int *)data)[1];
      for (i = 0; i < length; i ++)
        r += sprintf(page + r, "%02x ", ((unsigned char *)&atiNvramData)[offset + i]);
    }
    break;
  }
  r += sprintf(page + r, "\n");
  return (r < cnt)? r: 0;
#endif
}

static int proc_set_param(struct file *f, const char *buf, unsigned long cnt, void *data)
{
#if !defined(ATI_BSP_PERSONALITY)
    NVRAM_DATA *pNvramData;
    char input[32];

    int i = 0;
    int r = cnt;
    int offset  = ((int *)data)[0];
    int length  = ((int *)data)[1];

    if ((offset < 0) || (offset + length > sizeof(NVRAM_DATA)))
        return 0;

    if ((cnt > 32) || (copy_from_user(input, buf, cnt) != 0))
        return -EFAULT;

    for (i = 0; i < r; i ++)
    {
        if (!isxdigit(input[i]))
        {
            memmove(&input[i], &input[i + 1], r - i - 1);
            r --;
            i --;
        }
    }

    mutex_lock(&flashImageMutex);

    if (NULL != (pNvramData = readNvramData()))
    {
        str_to_num(input, ((char *)pNvramData) + offset, length);
        writeNvramDataCrcLocked(pNvramData);
    }
    else
    {
        cnt = 0;
    }

    mutex_unlock(&flashImageMutex);

    if (pNvramData)
        kfree(pNvramData);

    return cnt;
#else
    return 0;
#endif
}

#if defined(ATI_BSP_PERSONALITY)
// Call back function for returning GPIO state when /proc/gpio/ files are read.
// data argument contains the value of the environment variable associated with this GPIO,
// in int format. For example, data for the call back for AtiGpioBatteryLow is set to 0x98.
static int proc_get_gpio(char *page, char **start, off_t off, int cnt, int *eof, void *data)
{
    int r = 0;
	int gpioPinNum; 	// GPIO pin number with the MSB cleared
	int gpioInvLogic;	// 0: active-high, 0x80: active-low, and needs to be inverted
	int gpioVal = 0;

    // Is there enough space in the buffer?
    if (cnt < 2) {
        return 0;
    }
    gpioPinNum = (int)data;
    gpioInvLogic = gpioPinNum & 0x80;   // MSB set to 1 indicates active-low GPIO
    gpioPinNum &= ~0x80;                // mask off the MSB

	gpioVal = kerSysGetGpio(gpioPinNum);

	r = snprintf(page, 2, "%1d", gpioInvLogic ? ~gpioVal & 0x1 : gpioVal);

    return r;
}

/*
 * This function expects input in the form of: 
 *      echo "0" > /proc/gpio/AtiGpioFile 
 *      echo "1" > /proc/gpio/AtiGpioFile 
 *  
 *  "0" changes the value to inactive, leaves the direction unchanged.
 *  "1" changes the value to active, leaves the direction unchanged.
 */
static int proc_set_gpio(struct file *f, const char *buf, unsigned long cnt, void *data)
{
    char input[32];
    int gpioPinNum; 	// GPIO pin number with the MSB cleared
    int gpioInvLogic;	// 0: active-high, 0x80: active-low, and needs to be inverted

    if (cnt > 32)
        cnt = 32;

    if (copy_from_user(input, buf, cnt) != 0)
        return -EFAULT;

    gpioPinNum = (int)data;
    gpioInvLogic = gpioPinNum & 0x80;   // MSB set to 1 indicates active-low GPIO
    gpioPinNum &= ~0x80;                // mask off the MSB

    if (input[0] == '0')
    {
        kerSysSetGpio(gpioPinNum, gpioInvLogic ? 1 : 0);
    }
    if (input[0] == '1')
    {
        kerSysSetGpio(gpioPinNum, gpioInvLogic ? 0 : 1);
    }

    return cnt;
}

// Call back function for returning GPIO dirction when /proc/gpio/<gpioName>Dir files are read.
// data argument contains the value of the environment variable associated with this GPIO,
// in int format. For example, data for the call back for AtiGpioBatteryLow is set to 0x98.
// Page buffer is set to "in" for input and "out" for output.
static int proc_get_gpio_dir(char *page, char **start, off_t off, int cnt, int *eof, void *data)
{
    int r = 0;
	int gpioPinNum; 	// GPIO pin number with the MSB cleared
    int gpioDir = 0;

    // Is there enough space in the buffer?
    if (cnt < 4) {
        return 0;
    }
    gpioPinNum = (int)data;
    gpioPinNum &= ~0x80;                // mask off the MSB

    gpioDir = kerSysGetGpioDir(gpioPinNum);

	r = snprintf(page, 4, "%s", gpioDir == 1 ? "out": "in");

    return r;
}

/*
 * This function expects input in the form of: 
 *      echo "in" > /proc/gpio/AtiGpioDirFile
 *      echo "out" > /proc/gpio/AtiGpioDirFile
 */
static int proc_set_gpio_dir(struct file *f, const char *buf, unsigned long cnt, void *data)
{
    char input[32];
    int gpioPinNum; 	// GPIO pin number with the MSB cleared

    if (cnt > 32)
        cnt = 32;

    if (copy_from_user(input, buf, cnt) != 0)
        return -EFAULT;

    gpioPinNum = (int)data;
    //gpioPinNum &= ~0x80;                // mask off the MSB

    switch(cnt)
    {
        case 3:
            if ((input[0] == 'i' || input[0] == 'I') &&
                (input[1] == 'n' || input[1] == 'N'))
            {
                kerSysSetGpioDirToIn(gpioPinNum);
            }
            break;

        case 4:
            if ((input[0] == 'o' || input[0] == 'O') &&
                (input[1] == 'u' || input[1] == 'U') &&
                (input[2] == 't' || input[2] == 'T'))
            {
                kerSysSetGpioDirToOut(gpioPinNum);
            }
            break;

        default:
            return -EFAULT;
    }

    return cnt;
}

static int proc_get_led(char *page, char **start, off_t off, int cnt, int *eof, void *data)
{
    BOARD_LED_NAME i;
    int r = 0;
    r += sprintf(page + r,"%-20s:\t%-6s\t%-6s\t%-6s\t%-6s\t%s\n",
                 "   Name", "LED#",  "State", "Green", "Red", "Blink Cnt");
    for (i=0; i < kLedEnd; i++)
    {
        BOARD_LED_STATE state;
        char *name;
        int greenIO, greenInv, redIO, redInv, blinkCnt;
        if (!getLed(i, &state, &name, &greenIO, &redIO, &blinkCnt))
        {
            greenInv = (greenIO!=-1) && (greenIO!=0xff) && (greenIO&BP_ACTIVE_LOW);
            redInv = (redIO!=-1) && (redIO!=0xff) && (redIO&BP_ACTIVE_LOW);
            greenIO = greenIO&0x7f;
            redIO = redIO&0x7f;
            r += sprintf(page + r,"%-20s:\t%-6.2x\t%-6.2x\t%c%-6.2x\t%c%-6.2x\t%d\n", 
                         name, i, state, greenInv?'!':' ', greenIO, redInv?'!':' ', redIO, blinkCnt);
        }
    }
    return (r < cnt)? r: cnt;
}
#endif

/*
 * This function expect input in the form of:
 * echo "xxyy" > /proc/led
 * where xx is hex for the led number
 * and   yy is hex for the led state.
 * For example,
 *     echo "0301" > led
 * will turn on led 3
 */
static int proc_set_led(struct file *f, const char *buf, unsigned long cnt, void *data)
{
    char leddata[16];
    char input[32];
    int i;
    int r;
    int num_of_octets;

    if (cnt > 32)
        cnt = 32;

    if (copy_from_user(input, buf, cnt) != 0)
        return -EFAULT;

    r = cnt;

    for (i = 0; i < r; i ++)
    {
        if (!isxdigit(input[i]))
        {
            memmove(&input[i], &input[i + 1], r - i - 1);
            r --;
            i --;
        }
    }

    num_of_octets = r / 2;

    if (num_of_octets != 2)
        return -EFAULT;

    str_to_num(input, leddata, num_of_octets);
    kerSysLedCtrl (leddata[0], leddata[1]);
    return cnt;
}


#if defined(CONFIG_BCM96368)

#define DSL_PHY_PHASE_CNTL      ((volatile uint32* const) 0xb00012a8)
#define DSL_CPU_PHASE_CNTL      ((volatile uint32* const) 0xb00012ac)
#define MIPS_PHASE_CNTL         ((volatile uint32* const) 0xb00012b0)
#define DDR1_2_PHASE_CNTL       ((volatile uint32* const) 0xb00012b4)
#define DDR3_4_PHASE_CNTL       ((volatile uint32* const) 0xb00012b8)

// The direction bit tells the automatic counters to count up or down to the
// desired value.
#define PI_VALUE_WIDTH 14
#define PI_COUNT_UP    ( 1 << PI_VALUE_WIDTH )
#define PI_MASK        ( PI_COUNT_UP - 1 )

// Turn off sync mode.  Set bit 28 of CP0 reg 22 sel 5.
static void TurnOffSyncMode( void )
{
    uint32 value;

    value = __read_32bit_c0_register( $22, 5 ) | (1<<28);
    __write_32bit_c0_register( $22, 5, value );
    //    Print( "Sync mode %x\n", value );
    value = DDR->MIPSPhaseCntl;

    // Reset the PH_CNTR_CYCLES to 7.
    // Set the phase counter cycles (bits 16-19) back to 7.
    value &= ~(0xf<<16);
    value |= (7<<16);

    // Set the LLMB counter cycles back to 7.
    value &= ~(0xf<<24);
    value |= (7<<24);
    // Set the UBUS counter cycles back to 7.
    value &= ~(0xf<<28);
    value |= (7<<28);

    // Turn off the LLMB counter, which is what maintains sync mode.
    // Clear bit 21, which is LLMB_CNTR_EN.
    value &= ~(1 << 21);
    // Turn off UBUS LLMB CNTR EN
    value &= ~(1 << 23);

    DDR->MIPSPhaseCntl = value;

    // Reset the MIPS phase to 0.
    PI_lower_set( MIPS_PHASE_CNTL, 0 );

    //Clear Count Bit
    value &= ~(1 << 14);
    DDR->MIPSPhaseCntl = value;

}

// Write the specified value in the lower half of a PI control register.  Each
// 32-bit register holds two values, but they can't be addressed separately.
static void
PI_lower_set( volatile uint32  *PI_reg,
             int               newPhaseInt )
{
    uint32  oldRegValue;
    uint32  saveVal;
    int32   oldPhaseInt;
    int32   newPhase;
    uint32  newVal;
    int     equalCount      = 0;

    oldRegValue = *PI_reg;
    // Save upper 16 bits, which is the other PI value.
    saveVal     = oldRegValue & 0xffff0000;

    // Sign extend the lower PI value, and shift it down into the lower 16 bits.
    // This gives us a 32-bit signed value which we can compare to the newPhaseInt
    // value passed in.

    // Shift the sign bit to bit 31
    oldPhaseInt = oldRegValue << ( 32 - PI_VALUE_WIDTH );
    // Sign extend and shift the lower value into the lower 16 bits.
    oldPhaseInt = oldPhaseInt >> ( 32 - PI_VALUE_WIDTH );

    // Take the low 10 bits as the new phase value.
    newPhase = newPhaseInt & PI_MASK;

    // If our new value is larger than the old value, tell the automatic counter
    // to count up.
    if ( newPhaseInt > oldPhaseInt )
    {
        newPhase = newPhase | PI_COUNT_UP;
    }

    // Or in the value originally in the upper 16 bits.
    newVal  = newPhase | saveVal;
    *PI_reg = newVal;

    // Wait until we match several times in a row.  Only the low 4 bits change
    // while the counter is working, so we can get a false "done" indication
    // when we read back our desired value.
    do
    {
        if ( *PI_reg == newVal )
        {
            equalCount++;
        }
        else
        {
            equalCount = 0;
        }

    } while ( equalCount < 3 );

}

// Write the specified value in the upper half of a PI control register.  Each
// 32-bit register holds two values, but they can't be addressed separately.
static void
PI_upper_set( volatile uint32  *PI_reg,
             int               newPhaseInt )
{
    uint32  oldRegValue;
    uint32  saveVal;
    int32   oldPhaseInt;
    int32   newPhase;
    uint32  newVal;
    int     equalCount      = 0;

    oldRegValue = *PI_reg;
    // Save lower 16 bits, which is the other PI value.
    saveVal     = oldRegValue & 0xffff;

    // Sign extend the upper PI value, and shift it down into the lower 16 bits.
    // This gives us a 32-bit signed value which we can compare to the newPhaseInt
    // value passed in.

    // Shift the sign bit to bit 31
    oldPhaseInt = oldRegValue << ( 16 - PI_VALUE_WIDTH );
    // Sign extend and shift the upper value into the lower 16 bits.
    oldPhaseInt = oldPhaseInt >> ( 32 - PI_VALUE_WIDTH );

    // Take the low 10 bits as the new phase value.
    newPhase = newPhaseInt & PI_MASK;

    // If our new value is larger than the old value, tell the automatic counter
    // to count up.
    if ( newPhaseInt > oldPhaseInt )
    {
        newPhase = newPhase | PI_COUNT_UP;
    }

    // Shift the new phase value into the upper 16 bits, and restore the value
    // originally in the lower 16 bits.
    newVal = (newPhase << 16) | saveVal;
    *PI_reg = newVal;

    // Wait until we match several times in a row.  Only the low 4 bits change
    // while the counter is working, so we can get a false "done" indication
    // when we read back our desired value.
    do
    {
        if ( *PI_reg == newVal )
        {
            equalCount++;
        }
        else
        {
            equalCount = 0;
        }

    } while ( equalCount < 3 );

}

// Reset the DDR PI registers to the default value of 0.
static void ResetPiRegisters( void )
{
    volatile int delay;
    uint32 value;

    //Skip This step for now load_ph should be set to 0 for this anyways.
    //Print( "Resetting DDR phases to 0\n" );
    //PI_lower_set( DDR1_2_PHASE_CNTL, 0 ); // DDR1 - Should be a NOP.
    //PI_upper_set( DDR1_2_PHASE_CNTL, 0 ); // DDR2
    //PI_lower_set( DDR3_4_PHASE_CNTL, 0 ); // DDR3 - Must remain at 90 degrees for normal operation.
    //PI_upper_set( DDR3_4_PHASE_CNTL, 0 ); // DDR4

    // Need to have VDSL back in reset before this is done.
    // Disable VDSL Mip's
    GPIO->VDSLControl = GPIO->VDSLControl & ~VDSL_MIPS_RESET;
    // Disable VDSL Core
    GPIO->VDSLControl = GPIO->VDSLControl & ~(VDSL_CORE_RESET | 0x8);


    value = DDR->DSLCpuPhaseCntr;

    // Reset the PH_CNTR_CYCLES to 7.
    // Set the VDSL Mip's phase counter cycles (bits 16-19) back to 7.
    value &= ~(0xf<<16);
    value |= (7<<16);

    // Set the VDSL PHY counter cycles back to 7.
    value &= ~(0xf<<24);
    value |= (7<<24);
    // Set the VDSL AFE counter cycles back to 7.
    value &= ~(0xf<<28);
    value |= (7<<28);

    // Turn off the VDSL MIP's PHY auto counter
    value &= ~(1 << 20);
    // Clear bit 21, which is VDSL PHY CNTR_EN.
    value &= ~(1 << 21);
    // Turn off the VDSL AFE auto counter
    value &= ~(1 << 22);

    DDR->DSLCpuPhaseCntr = value;

    // Reset the VDSL MIPS phase to 0.
    PI_lower_set( DSL_PHY_PHASE_CNTL, 0 ); // VDSL PHY - should be NOP
    PI_upper_set( DSL_PHY_PHASE_CNTL, 0 ); // VDSL AFE - should be NOP
    PI_lower_set( DSL_CPU_PHASE_CNTL, 0 ); // VDSL MIP's

    //Clear Count Bits for DSL CPU
    value &= ~(1 << 14);
    DDR->DSLCpuPhaseCntr = value;
    //Clear Count Bits for DSL Core
    DDR->DSLCorePhaseCntl &= ~(1<<30);
    DDR->DSLCorePhaseCntl &= ~(1<<14);
    // Allow some settle time.
    delay = 100;
    while (delay--);

    printk("\n****** DDR->DSLCorePhaseCntl=%lu ******\n\n", (unsigned long)
        DDR->DSLCorePhaseCntl);

    // Turn off the automatic counters.
    // Clear bit 20, which is PH_CNTR_EN.
    DDR->MIPSPhaseCntl &= ~(1<<20);
    // Turn Back UBUS Signals to reset state
    DDR->UBUSPhaseCntl = 0x00000000;
    DDR->UBUSPIDeskewLLMB0 = 0x00000000;
    DDR->UBUSPIDeskewLLMB1 = 0x00000000;

}

static void ChipSoftReset(void)
{
    TurnOffSyncMode();
    ResetPiRegisters();
}
#endif


/***************************************************************************
 * Function Name: kerSysGetUbusFreq
 * Description  : Chip specific computation.
 * Returns      : the UBUS frequency value in MHz.
 ***************************************************************************/
unsigned long kerSysGetUbusFreq(unsigned long miscStrapBus)
{
   unsigned long ubus = UBUS_BASE_FREQUENCY_IN_MHZ;

#if defined(CONFIG_BCM96362)
   /* Ref RDB - 6362 */
   switch (miscStrapBus) {

      case 0x4 :
      case 0xc :
      case 0x14:
      case 0x1c:
      case 0x15:
      case 0x1d:
         ubus = 100;
         break;
      case 0x2 :
      case 0xa :
      case 0x12:
      case 0x1a:
         ubus = 96;
         break;
      case 0x1 :
      case 0x9 :
      case 0x11:
      case 0xe :
      case 0x16:
      case 0x1e:
         ubus = 200;
         break;
      case 0x6:
         ubus = 183;
         break;
      case 0x1f:
         ubus = 167;
         break;
      default:
         ubus = 160;
         break;
   }
#endif

   return (ubus);

}  /* kerSysGetUbusFreq */

#if defined(ATI_PRODUCT_CONFIG) && (defined(CONFIG_ATI_SFP) || defined(CONFIG_BCM_I2C_BUS))
/** Look for AtiGpioSfpI2cClk, AtiGpioSfpI2cDat and AtiGpioSfpPresent
 *  definitions in NVRAM data.
 *  
 * @param scl_pin [out] GPIO number to use for I2C SCL signal 
 * @param scl_pin_found [out] 1 = GPIO number of SCL signal was found, 0 = not found 
 * @param sda_pin [out] GPIO number to use for I2C SDA signal 
 * @param sda_pin_found [out] 1 = GPIO number of SDA signal was found, 0 = not found 
 * @param pres_pin [out] GPIO number to use for SFP present signal 
 * @param pres_pin_found [out] 1 = GPIO number of SFP present signal was found, 
 * 0 = not found 
 * 
 * @return int: 0 if no errors where encountered, -1 otherwise.
 */
static int get_i2c_sfp_gpio_numbers(unsigned int *scl_pin, 
                                    unsigned int *scl_pin_found,
                                    unsigned int *sda_pin,
                                    unsigned int *sda_pin_found,
                                    unsigned int *pres_pin,
                                    unsigned int *pres_pin_found)
{
  int envCount = 0;
  *scl_pin_found = 0;
  *sda_pin_found = 0;
  *pres_pin_found = 0;

  while (!(NvRamLookUpTable[envCount][0] == 0 && NvRamLookUpTable[envCount][1] == 0 && NvRamLookUpTable[envCount][2] == 0) &&
         !(*scl_pin_found && *sda_pin_found && *pres_pin_found))
  {
      if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiGpioSfpI2cClk", 16) == 0) {
          if (!strict_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 16, (unsigned long *)scl_pin))
          {
              *scl_pin_found = 1;
              envCount++;
              continue;
          }
          else
          {
              return -1;
          }
      }

      if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiGpioSfpI2cDat", 16) == 0) {
          if (!strict_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 16, (unsigned long *)sda_pin))
          {
              *sda_pin_found = 1;
              envCount++;
              continue;
          }
          else
          {
              return -1;
          }
      }

      if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiGpioSfpPresent", 17) == 0) {
          if (!strict_strtoul(&atiNvramData.buffer[NvRamLookUpTable[envCount][2]], 16, (unsigned long *)pres_pin))
          {
              /* 
               * SFP present is an active-low signal, which is indicated by bit 8 == 1, 
               * so, subtract 128 to get the actual GPIO number. 
               */
              if (*pres_pin >= 128)
              {
                  *pres_pin -= 128;
              }
              *pres_pin_found = 1;
              envCount++;
              continue;
          }
          else
          {
              return -1;
          }
      }

      envCount++;
  }

  return 0;
}

/**
 * Kernel workqueue handler for SFP tasks. It runs periodically using the schedule_delayed_work(). 
 *  
 * Sample SFP presence GPIO signal. If the state of the presence signal changes, add or 
 * delete the SFP to/from the system.
 */
static void sfp_work_queue_handler(struct work_struct *work)
{
    static int prevState = 1;
    sfp_info_t *sfp_info = container_of(work, sfp_info_t, sfp_work.work);
    int curState = 0;

    curState = kerSysGetGpio((unsigned short)sfp_info->sfp_pres_gpio);

    if (curState != prevState)
    {
        prevState = curState;
        if (curState == 0)
        {
            if (0 != atlat_sfp_add_sfp(sfp_info->i2c_adap, sfp_info->sfp_board_index, sfp_info->sfp_port_index))
            {
                printk("SFP insertion detected, but SFP could not be added to the the system.\n");
            }
        }
        else
        {
            if (0 != atlat_sfp_delete_sfp(sfp_info->sfp_board_index, sfp_info->sfp_port_index))
            {
                printk("SFP removal detected, but SFP could not be deleted from the system.\n");
            }
        }
    }

    schedule_delayed_work(&sfp_info->sfp_work, sfp_info->poll_interval);
}

/*
 * AtiEthSwitchMap parameters passed from the bootloader defines a mapping between Ethernet 
 * switch port numbers and virtual Linux (eth) interface numbers. 
 *  
 * For example, "set AtiEthSwitchMap 01 00 05 02 08 0A 04 07" sets up the following map: 
 *    eth0 <-> switch port 1
 *    eth1 <-> switch port 0
 *    eth2 <-> switch port 5
 *    eth3 <-> switch port 2
 *    eth4 <-> switch port 8 (this happened to be an external switch)
 *    eth5 <-> switch port 4 (this is the active Ethernet WAN port)
 *    eth6 <-> switch port 7 (this is the GPON port)
 *  
 * So, to get eth port number associated with the SFP port, we look for port 4 in the mask, 
 * and return its index in the mask. 
 *  
 * Of course, this could be smarter to allow SFP interface on other ports...
 */
static int get_sfp_vport_number_from_AtiEthSwitchMap(void)
{
    int envCount = 0;
    int portIndex=0;
    int found = 0;

    while (!(NvRamLookUpTable[envCount][0] == 0 && NvRamLookUpTable[envCount][1] == 0 && NvRamLookUpTable[envCount][2] == 0) &&
           !found)
    {
        if (strncmp(&atiNvramData.buffer[NvRamLookUpTable[envCount][1]], "AtiEthSwitchMap", 15) == 0)
        {
            char *pValue=&atiNvramData.buffer[NvRamLookUpTable[envCount][2]];
            char *pEndValue;

            do
            {
                int value;
                
                value = simple_strtoul(pValue, &pEndValue, 16);
                if (value == 4)
                {
                    break;
                }

                pValue = pEndValue + 1;
                portIndex++;
            } while (*pValue!='\0' && portIndex < 16);
            found = 1;
        }
        envCount++;
    }
    return portIndex;
}
#endif

/***************************************************************************
 * Function Name: kerSysGetChipId
 * Description  : Map id read from device hardware to id of chip family
 *                consistent with  BRCM_CHIP
 * Returns      : chip id of chip family
 ***************************************************************************/
int kerSysGetChipId() { 
        int r;
        int t;
        r = (int) ((PERF->RevID & CHIP_ID_MASK) >> CHIP_ID_SHIFT);
        /* Force BCM681x variants to be BCM6816 or BCM6818 correctly) */
        if( (r & 0xfff0) == 0x6810 )
        {
            if ((r == 0x6811) || (r == 0x6815) || (r == 0x6817)) {
                r = 0x6818;
            } else {
                r = 0x6816;
                t = (int) (PERF->RevID & REV_ID_MASK);
                if ((t & 0xf0) == 0xA0)
                {
                    r = 0x6818;
                }
#if defined(CONFIG_BRCM_6818_ON_6816)
                r = 0x6818;
#endif
            }
        }

        /* Force BCM6369 to be BCM6368) */
        if( (r & 0xfffe) == 0x6368 )
            r = 0x6368;

        /* Force 6821 and 6822 to be BCM6828 */
        if ((r == 0x6821) || (r == 0x6822))
            r = 0x6828;

        /* Force BCM63168, BCM63169, and BCM63269 to be BCM63268) */
        if( ( (r & 0xffffe) == 0x63168 )
          || ( (r & 0xffffe) == 0x63268 ))
            r = 0x63268;

        return(r);
}

/***************************************************************************
 * Function Name: kerSysGetDslPhyEnable
 * Description  : returns true if device should permit Phy to load
 * Returns      : true/false
 ***************************************************************************/
int kerSysGetDslPhyEnable() {
        int id;
        int r = 1;
        id = (int) ((PERF->RevID & CHIP_ID_MASK) >> CHIP_ID_SHIFT);
        if ((id == 0x63169) || (id == 0x63269)) {
	    r = 0;
        }
        return(r);
}

/***************************************************************************
 * Function Name: kerSysGetChipName
 * Description  : fills buf with the human-readable name of the device
 * Returns      : pointer to buf
 ***************************************************************************/
char *kerSysGetChipName(char *buf, int n) {
	return(UtilGetChipName(buf, n));
}

/***************************************************************************
* MACRO to call driver initialization and cleanup functions.
***************************************************************************/
module_init( brcm_board_init );
module_exit( brcm_board_cleanup );

EXPORT_SYMBOL(dumpaddr);
EXPORT_SYMBOL(kerSysGetChipId);
EXPORT_SYMBOL(kerSysGetChipName);
EXPORT_SYMBOL(kerSysMacAddressNotifyBind);
EXPORT_SYMBOL(kerSysGetMacAddressType);
EXPORT_SYMBOL(kerSysGetMacAddress);
EXPORT_SYMBOL(kerSysReleaseMacAddress);
EXPORT_SYMBOL(kerSysGetGponSerialNumber);
EXPORT_SYMBOL(kerSysGetGponPassword);
EXPORT_SYMBOL(kerSysGetSdramSize);
EXPORT_SYMBOL(kerSysGetDslPhyEnable);
EXPORT_SYMBOL(kerSysSetOpticalPowerValues);
EXPORT_SYMBOL(kerSysGetOpticalPowerValues);
#if defined(CONFIG_BCM96368)
EXPORT_SYMBOL(kerSysGetSdramWidth);
#endif
EXPORT_SYMBOL(kerSysLedCtrl);
EXPORT_SYMBOL(kerSysRegisterDyingGaspHandler);
EXPORT_SYMBOL(kerSysDeregisterDyingGaspHandler);
EXPORT_SYMBOL(kerSysSendtoMonitorTask);
EXPORT_SYMBOL(kerSysGetWlanSromParams);
EXPORT_SYMBOL(kerSysGetAfeId);
EXPORT_SYMBOL(kerSysGetUbusFreq);
#if defined(CONFIG_BCM96816) || defined(CONFIG_BCM96818)
EXPORT_SYMBOL(kerSysBcmSpiSlaveRead);
EXPORT_SYMBOL(kerSysBcmSpiSlaveReadReg32);
EXPORT_SYMBOL(kerSysBcmSpiSlaveWrite);
EXPORT_SYMBOL(kerSysBcmSpiSlaveWriteReg32);
EXPORT_SYMBOL(kerSysBcmSpiSlaveWriteBuf);
#endif
EXPORT_SYMBOL(BpGetBoardId);
EXPORT_SYMBOL(BpGetBoardIds);
EXPORT_SYMBOL(BpGetGPIOverlays);
EXPORT_SYMBOL(BpGetFpgaResetGpio);
EXPORT_SYMBOL(BpGetEthernetMacInfo);
#if defined(CONFIG_BCM963268) && (CONFIG_BCM_EXT_SWITCH)
EXPORT_SYMBOL(BpGetPortConnectedToExtSwitch);
#endif
EXPORT_SYMBOL(BpGetRj11InnerOuterPairGpios);
EXPORT_SYMBOL(BpGetRtsCtsUartGpios);
EXPORT_SYMBOL(BpGetAdslLedGpio);
EXPORT_SYMBOL(BpGetAdslFailLedGpio);
EXPORT_SYMBOL(BpGetWanDataLedGpio);
EXPORT_SYMBOL(BpGetWanErrorLedGpio);
EXPORT_SYMBOL(BpGetVoipLedGpio);
EXPORT_SYMBOL(BpGetPotsLedGpio);
EXPORT_SYMBOL(BpGetVoip2FailLedGpio);
EXPORT_SYMBOL(BpGetVoip2LedGpio);
EXPORT_SYMBOL(BpGetVoip1FailLedGpio);
EXPORT_SYMBOL(BpGetVoip1LedGpio);
EXPORT_SYMBOL(BpGetDectLedGpio);
EXPORT_SYMBOL(BpGetMoCALedGpio);
EXPORT_SYMBOL(BpGetMoCAFailLedGpio);
EXPORT_SYMBOL(BpGetWirelessSesExtIntr);
EXPORT_SYMBOL(BpGetWirelessSesLedGpio);
EXPORT_SYMBOL(BpGetWirelessFlags);
EXPORT_SYMBOL(BpGetWirelessPowerDownGpio);
EXPORT_SYMBOL(BpUpdateWirelessSromMap);
EXPORT_SYMBOL(BpGetSecAdslLedGpio);
EXPORT_SYMBOL(BpGetSecAdslFailLedGpio);
EXPORT_SYMBOL(BpGetDslPhyAfeIds);
EXPORT_SYMBOL(BpGetExtAFEResetGpio);
EXPORT_SYMBOL(BpGetExtAFELDPwrGpio);
EXPORT_SYMBOL(BpGetExtAFELDModeGpio);
EXPORT_SYMBOL(BpGetIntAFELDPwrGpio);
EXPORT_SYMBOL(BpGetIntAFELDModeGpio);
EXPORT_SYMBOL(BpGetAFELDRelayGpio);
EXPORT_SYMBOL(BpGetExtAFELDDataGpio);
EXPORT_SYMBOL(BpGetExtAFELDClkGpio);
EXPORT_SYMBOL(BpGetUart2SdoutGpio);
EXPORT_SYMBOL(BpGetUart2SdinGpio);
EXPORT_SYMBOL(BpGet6829PortInfo);
EXPORT_SYMBOL(BpGetEthSpdLedGpio);
EXPORT_SYMBOL(BpGetLaserDisGpio);
EXPORT_SYMBOL(BpGetLaserTxPwrEnGpio);
EXPORT_SYMBOL(BpGetVregSel1P2);
EXPORT_SYMBOL(BpGetGponOpticsType);
EXPORT_SYMBOL(BpGetDefaultOpticalParams);
EXPORT_SYMBOL(BpGetI2cGpios);
EXPORT_SYMBOL(BpGetMiiOverGpioFlag);
EXPORT_SYMBOL(BpGetSwitchPortMap);
#if defined (CONFIG_BCM_ENDPOINT_MODULE)
EXPORT_SYMBOL(BpGetVoiceBoardId);
EXPORT_SYMBOL(BpGetVoiceBoardIds);
EXPORT_SYMBOL(BpGetVoiceParms);
#endif
#if defined (CONFIG_BCM_AVS_PWRSAVE)
EXPORT_SYMBOL(kerSysBcmEnableAvs);
EXPORT_SYMBOL(kerSysBcmAvsEnabled);
#endif

#if defined(CONFIG_EPON_SDK)
EXPORT_SYMBOL(BpGetNumFePorts);
EXPORT_SYMBOL(BpGetNumGePorts);
EXPORT_SYMBOL(BpGetNumVoipPorts);
#endif

