/*
 * Bitbanging I2C bus driver using the ATI-modified bcm963xx GPIO.
 *
 * Copyright (c) 2011 Allied Telesis Holdings K. K.
 * 
 * This code is based on the i2c-gpio driver.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/i2c.h>
#include <linux/i2c-algo-bit.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <i2c-ati-gpio.h>
#include <board.h>

/* Toggle SDA by changing the direction of the pin */
static void i2c_ati_gpio_setsda_dir(void *data, int state)
{
	struct i2c_ati_gpio_platform_data *pdata = data;

	if (state)
    {
        kerSysSetGpioDirToIn(pdata->sda_pin);
    }
	else
    {
        kerSysSetGpioDirToOut(pdata->sda_pin);
        kerSysSetGpio(pdata->sda_pin, kGpioInactive);
    }
}

/*
 * Toggle SDA by changing the output value of the pin. This is only
 * valid for pins configured as open drain (i.e. setting the value
 * high effectively turns off the output driver.)
 */
static void i2c_ati_gpio_setsda_val(void *data, int state)
{
	struct i2c_ati_gpio_platform_data *pdata = data;

    if (state)
        kerSysSetGpio(pdata->sda_pin, kGpioActive);
    else
        kerSysSetGpio(pdata->sda_pin, kGpioInactive);
}

/* Toggle SCL by changing the direction of the pin. */
static void i2c_ati_gpio_setscl_dir(void *data, int state)
{
	struct i2c_ati_gpio_platform_data *pdata = data;

	if (state)
        kerSysSetGpioDirToIn(pdata->scl_pin);
	else
        kerSysSetGpioDirToOut(pdata->scl_pin);
}

/*
 * Toggle SCL by changing the output value of the pin. This is used
 * for pins that are configured as open drain and for output-only
 * pins. The latter case will break the i2c protocol, but it will
 * often work in practice.
 */
static void i2c_ati_gpio_setscl_val(void *data, int state)
{
	struct i2c_ati_gpio_platform_data *pdata = data;

    if (state)
        kerSysSetGpio(pdata->scl_pin, kGpioActive);
    else
        kerSysSetGpio(pdata->scl_pin, kGpioInactive);
}

static int i2c_ati_gpio_getsda(void *data)
{
	struct i2c_ati_gpio_platform_data *pdata = data;

    return kerSysGetGpio(pdata->sda_pin);
}

static int i2c_ati_gpio_getscl(void *data)
{
	struct i2c_ati_gpio_platform_data *pdata = data;

    return kerSysGetGpio(pdata->scl_pin);
}

static int __devinit i2c_ati_gpio_probe(struct platform_device *pdev)
{
	struct i2c_ati_gpio_platform_data *pdata;
	struct i2c_algo_bit_data *bit_data;
	struct i2c_adapter *adap;
	int ret;

	pdata = pdev->dev.platform_data;
	if (!pdata)
		return -ENXIO;

	ret = -ENOMEM;
	adap = kzalloc(sizeof(struct i2c_adapter), GFP_KERNEL);
	if (!adap)
		goto err_alloc_adap;
	bit_data = kzalloc(sizeof(struct i2c_algo_bit_data), GFP_KERNEL);
	if (!bit_data)
		goto err_alloc_bit_data;

	if (pdata->sda_is_open_drain) {
		bit_data->setsda = i2c_ati_gpio_setsda_val;
	} else {
        kerSysSetGpioDirToIn(pdata->sda_pin);
		bit_data->setsda = i2c_ati_gpio_setsda_dir;
	}

	if (pdata->scl_is_open_drain || pdata->scl_is_output_only) {
        kerSysSetGpioDirToOut(pdata->scl_pin);
        kerSysSetGpio(pdata->scl_pin, kGpioActive);
		bit_data->setscl = i2c_ati_gpio_setscl_val;
	} else {
		bit_data->setscl = i2c_ati_gpio_setscl_dir;
	}

	if (!pdata->scl_is_output_only)
    {
		bit_data->getscl = i2c_ati_gpio_getscl;
    }
	bit_data->getsda = i2c_ati_gpio_getsda;

	if (pdata->udelay)
    {
		bit_data->udelay = pdata->udelay;
    }
	else if (pdata->scl_is_output_only)
		bit_data->udelay = 50;			/* 10 kHz */
	else
		bit_data->udelay = 5;			/* 100 kHz */

	if (pdata->timeout)
		bit_data->timeout = pdata->timeout;
	else
		bit_data->timeout = HZ / 10;		/* 100 ms */

	bit_data->data = pdata;

	adap->owner = THIS_MODULE;
	snprintf(adap->name, sizeof(adap->name), "i2c-ati-gpio%d", pdev->id);
	adap->algo_data = bit_data;
	adap->class = I2C_CLASS_SPD;
	adap->dev.parent = &pdev->dev;

	/*
	 * If "dev->id" is negative we consider it as zero.
	 * The reason to do so is to avoid sysfs names that only make
	 * sense when there are multiple adapters.
	 */
	adap->nr = (pdev->id != -1) ? pdev->id : 0;
	ret = i2c_bit_add_numbered_bus(adap);
	if (ret)
		goto err_add_bus;

	platform_set_drvdata(pdev, adap);

	dev_info(&pdev->dev, "using pins %u (SDA) and %u (SCL%s)\n",
		 pdata->sda_pin, pdata->scl_pin,
		 pdata->scl_is_output_only
		 ? ", no clock stretching" : "");

	return 0;

err_add_bus:
	kfree(bit_data);
err_alloc_bit_data:
	kfree(adap);
err_alloc_adap:
	return ret;
}

static int __devexit i2c_ati_gpio_remove(struct platform_device *pdev)
{
	struct i2c_ati_gpio_platform_data *pdata;
	struct i2c_adapter *adap;

	adap = platform_get_drvdata(pdev);
	pdata = pdev->dev.platform_data;

	i2c_del_adapter(adap);
	kfree(adap->algo_data);
	kfree(adap);

	return 0;
}

static struct platform_driver i2c_ati_gpio_driver = {
	.driver		= {
		.name	= "i2c-ati-gpio",
		.owner	= THIS_MODULE,
	},
	.probe		= i2c_ati_gpio_probe,
	.remove		= __devexit_p(i2c_ati_gpio_remove),
};

static int __init i2c_ati_gpio_init(void)
{
	int ret;

	ret = platform_driver_register(&i2c_ati_gpio_driver);
	if (ret)
		printk(KERN_ERR "i2c-ati-gpio: probe failed: %d\n", ret);

	return ret;
}

/* 
 * To cause this driver to be loaded early... 
 * This is necessary so that the bus driver is loaded before 
 * device drivers. 
 */
subsys_initcall(i2c_ati_gpio_init);

static void __exit i2c_ati_gpio_exit(void)
{
	platform_driver_unregister(&i2c_ati_gpio_driver);
}
module_exit(i2c_ati_gpio_exit);

MODULE_DESCRIPTION("Platform-dependent bitbanging I2C driver for bcm963xx using ATI LED GPIO");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:i2c-ati-gpio");
