/*
 * atlat_sfp.h - Driver for the SFPs - class of interface
 * Graeme K. Campbell<graeme.campbell@alliedtelesyn.co.nz>
 *
 * Copyright (C) 2010 Allied Telesis Labs NZ
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#ifndef _ATLAT_SFP_H_
#define _ATLAT_SFP_H_

#include <linux/i2c.h>

#define SFP_I2C_ADDRESS		0x50
#define SFP_PHY_I2C_ADDRESS	0x56
#define SFP_STARTUP_RETRIES     16
#define SFP_DDM_SUPPORT_MASK    0x40
#define SFP_DIAG_I2C_ADDRESS	0x51
#define SFP_DIAG_RETRIES     4

/* 	SFP MSA Serial Info definition, data array sizes. */
#define SFP_MSA_SERIAL_TXER_BYTES		8
#define SFP_MSA_SERIAL_VENDOR_NAME_BYTES	16
#define SFP_MSA_SERIAL_VENDOR_OUI_BYTES		3
#define SFP_MSA_SERIAL_VENDOR_PN_BYTES		16
#define SFP_MSA_SERIAL_VENDOR_REV_BYTES		4
#define SFP_MSA_SERIAL_WAVELENGTH_BYTES		2
#define SFP_MSA_SERIAL_OPTIONS_BYTES		2
#define SFP_MSA_SERIAL_VENDOR_SN_BYTES		16
#define SFP_MSA_SERIAL_DATE_CODE_BYTES		8
#define SFP_MSA_SERIAL_VENDOR_SPECIFIC_BYTES	32

#define SFP_MSA_SERIAL_INFO_READ_BLOCK_SIZE	128

/* 	SFP MSA Diagnostics Info definition, data array sizes. */
#define SFP_MSA_DIAGNOSTIC_3_BYTES		3
#define SFP_MSA_DIAGNOSTIC_4_BYTES		4
#define SFP_MSA_DIAGNOSTIC_16_BYTES		16

#define SFP_MSA_DIAG_THRESHOLDS_MAX	10
#define SFP_MSA_DIAG_THRESHOLD_FIELDS	20
#define SFP_MSA_DIAG_CAL_FIELDS_32		5
#define SFP_MSA_DIAG_CAL_FIELDS_16		2

/**
 * \struct sfpMsaSerialIdS is a container for info retrieved from an SFP serial EEPROM
 *	in a format defined by the SFP Multi-source Agreement.
 */
typedef struct xcvrSfpMsaSerialIdS
{
	__u8		id;
	__u8		extId;
	__u8		connector;
	__u8		txer[SFP_MSA_SERIAL_TXER_BYTES];
	__u8		encoding;
	__u8		br;
	__u8		reserved1;
	__u8		length9u1;
	__u8		length9u2;
	__u8		length50u;
	__u8		length62dot5u;
	__u8		lengthCopper;
	__u8		reserved2;
	__u8		vendorName[SFP_MSA_SERIAL_VENDOR_NAME_BYTES];
	__u8		reserved3;
	__u8		vendorOui[SFP_MSA_SERIAL_VENDOR_OUI_BYTES];
	__u8		vendorPn[SFP_MSA_SERIAL_VENDOR_PN_BYTES];
	__u8		vendorRev[SFP_MSA_SERIAL_VENDOR_REV_BYTES];
	__u8		wavelength[SFP_MSA_SERIAL_WAVELENGTH_BYTES];
#if defined(ATI_PRODUCT_CONFIG)
	__u8		ativalidation;
#else
	__u8		reserved4;
#endif
	__u8		ccBase;
	__u8		options[SFP_MSA_SERIAL_OPTIONS_BYTES];
	__u8		maxBr;
	__u8		minBr;
	__u8		vendorSn[SFP_MSA_SERIAL_VENDOR_SN_BYTES];
	__u8		dateCode[SFP_MSA_SERIAL_DATE_CODE_BYTES];
	__u8		dmtype;
	__u8		reserved5[2];
	__u8		ccExt;
	__u8		vendorSpecificData[SFP_MSA_SERIAL_VENDOR_SPECIFIC_BYTES];
} sfpMsaSerialIdT;

/**
 * \struct xcvrSfpMsaDiagnosticsS is a container for info retrieved from an SFP
 *  serial EEPROM diagnostics in a format defined by the SFP Multi-source Agreement.
 */
typedef struct xcvrSfpMsaDiagnosticsS
{
//	Alarm and warning thresholds
	__u16		alarmThresholds[SFP_MSA_DIAG_THRESHOLDS_MAX];
	__u16		warningThresholds[SFP_MSA_DIAG_THRESHOLDS_MAX];
	__u8		reserved1[SFP_MSA_DIAGNOSTIC_16_BYTES];
//	Calibration constants
	__u32		calRxPwr[SFP_MSA_DIAG_CAL_FIELDS_32];
	__u16		calTxLas[SFP_MSA_DIAG_CAL_FIELDS_16];
	__u16		calTxPwr[SFP_MSA_DIAG_CAL_FIELDS_16];
	__u16		calTemp[SFP_MSA_DIAG_CAL_FIELDS_16];
	__u16		calVolt[SFP_MSA_DIAG_CAL_FIELDS_16];
	__u8		reserved2[SFP_MSA_DIAGNOSTIC_3_BYTES];
	__u8		checkSum;
//	Diagnostic and control constants
	__u16		temperature;
	__u16		vcc;
	__u16		txBias;
	__u16		txPower;
	__u16		rxPower;
	__u8		reserved3[SFP_MSA_DIAGNOSTIC_4_BYTES];
	__u8		statusAndControl;
	__u8		reserved4;
//	Alarm and warning flagss
	__u16		alarmFlags;
	__u8		reserved5;
	__u8		reserved6;
	__u16		warningFlags;
//  Extended module control/status
	__u8		extndCtrl;
	__u8		reserved7;
	__u8		vendorSpecific[SFP_MSA_DIAGNOSTIC_4_BYTES];
// User accessible EEPROM locations - not implemented
} sfpMsaDiagnosticsT;

extern int atlat_sfp_init(void);
extern void atlat_sfp_exit(void);
extern int atlat_sfp_add_sfp(struct i2c_adapter *adap, int boardIndex, int portIndex);
extern int atlat_sfp_add_sfp_with_data(int boardIndex, int portIndex, const u8 *data);
extern int atlat_sfp_add_sfp_diagnostics_data(int boardIndex, int portIndex, u8 *data);
extern int atlat_sfp_delete_sfp(int boardIndex, int portIndex);
extern void atlat_sfp_delete_board_sfps(int boardIndex);
extern int atlat_sfp_present(int boardIndex, int portIndex);
#endif /* _ATLAT_SFP_H_ */

