/*
 * Copyright (C) 2010 Allied Telesis Labs NZ
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/delay.h>
#include <linux/i2c.h>
#if !defined(ATI_PRODUCT_CONFIG)
#include "atlat_i2c.h"
#endif

#if defined(ATI_PRODUCT_CONFIG)
#define I2C_RETRY_COUNT		5
#endif

int atlat_i2c_transaction_retry(struct i2c_adapter *adapter, int slaveId,
				u16 numBytesToRx, u16 numBytesToTx,
				u8 * pBytesToRx, u8 * pBytesToTx)
{
	struct i2c_msg msg;
	struct i2c_adapter *adap;
	int ret;
	int i;
	unsigned int nr = adapter->nr;

	msg.addr = slaveId;
	msg.flags = 0;
	msg.len = numBytesToTx;
	msg.buf = (char *)pBytesToTx;

	for (i = 0; i < I2C_RETRY_COUNT; i++) {
		adap = i2c_get_adapter(nr);
		if (adap == NULL) {
			return -1;
		}

		ret = i2c_transfer(adap, &msg, 1);
		/* If everything went ok (i.e. 1 msg transmitted), return #bytes
		   receieved, else error code. */
		if (ret == 1) {
			break;
		}
		(void)msleep_interruptible(5);
	}

	if (i >= I2C_RETRY_COUNT) {
		pr_debug("%s:%u - Retry count exceeded, RX retry %d ret %x\n",
		    __func__, __LINE__, i, ret);
	}

	/* If everything went ok (i.e. 1 msg transmitted), return #bytes
	   transmitted, else error code. */
	if (ret != 1) {
		pr_debug("%s:%u - RX ret %x\n", __func__, __LINE__, ret);
		return -1;
	}

	if (numBytesToRx && pBytesToRx != NULL) {
		msg.addr = slaveId;
		msg.flags = 0;
		msg.flags |= I2C_M_RD;
		msg.len = numBytesToRx;
		msg.buf = (char *)pBytesToRx;
		for (i = 0; i < I2C_RETRY_COUNT; i++) {
			adap = i2c_get_adapter(nr);
			if (adap == NULL) {
				return -1;
			}
			ret = i2c_transfer(adap, &msg, 1);
			/* If everything went ok (i.e. 1 msg transmitted), return #bytes
			   receieved, else error code. */
			if (ret == 1) {
				return 0;
			}
			(void)msleep_interruptible(5);
		}

		if (i >= I2C_RETRY_COUNT) {
			printk(KERN_INFO "%s:%u - RX retry %d ret %x\n",
			       __func__, __LINE__, i, ret);
		}

		pr_debug("%s - RX ret %x\n", __func__, ret);

		if (ret != 1) {
			printk(KERN_WARNING "%s:%u - RX ret %x\n", __func__,
			       __LINE__, ret);
			return -1;
		}
	}

	pr_debug("%s - read %d bytes from slave ID %x\n", __func__, numBytesToRx,
	    slaveId);
	return 0;
}

#if 0
int atlat_i2c_smbus_xfer_retry(struct i2c_adapter *adapter, u16 addr,
			       unsigned short flags, char read_write,
			       u8 command, int size, union i2c_smbus_data *data)
{
	int i;
	unsigned int nr = adapter->nr;
	struct i2c_adapter *adap;

	for (i = 0; i < I2C_RETRY_COUNT; i++) {
		adap = i2c_get_adapter(nr);
		if (adap == NULL) {
			return -1;
		}

		if (0 ==
		    i2c_smbus_xfer(adap, addr, flags, read_write, command, size,
				   data)) {
			return 0;
		}
		(void)msleep_interruptible(500);
	}
	printk(KERN_ERR "%s:%u I2C bad transfer address %02x value %04x\n",
	       __func__, __LINE__, (unsigned int)addr,
	       (unsigned int)data->word);
	return -1;
}
#endif

