/*
 * atlat_sfp.c - Driver for the SFPs - class of interface
 * Graeme K. Campbell<graeme.campbell@alliedtelesyn.co.nz>
 * Copyright (C) 2010 Allied Telesis Labs NZ
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#undef DEBUG
//#define DEBUG 1

#include <linux/module.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/slab.h>
#include <linux/jiffies.h>
#if !defined(ATI_PRODUCT_CONFIG)
#include <linux/boardinfo.h>
#endif
#include <linux/err.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#if !defined(ATI_PRODUCT_CONFIG)
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/ATL/epi3.h>
#include <linux/ATL/atl_pluggables.h>
#include "atlat_pci_hp.h"
#include "atlat_sfp.h"
#include "atlat_i2c.h"
#endif
#if defined(ATI_PRODUCT_CONFIG)
#include <atlat_sfp.h>
#include <atl_pluggables.h>
#include <bcmnetlink.h>
#include "board.h"
extern int atlat_i2c_transaction_retry( struct i2c_adapter *adap, int slaveId, u16 numBytesToRx, 
                                        u16 numBytesToTx, u8 *pBytesToRx, u8 *pBytesToTx);
#endif

#define SFP_ID_PREFIX "sfp"
#define SFP_ID_FORMAT SFP_ID_PREFIX "%d"

typedef struct _sfpBoard
{
	struct list_head    list;
	struct device       dev;
	struct device      *cdev;
	int                 valid;	/* !=0 if following fields are valid */
	int                 boardIndex;
	int                 portIndex;
	struct i2c_adapter *adap;
	sfpMsaSerialIdT     sfpMsaSerialId;
	sfpMsaDiagnosticsT  sfpMsaDiagnostics;
	unsigned long       dm_last_updated;
	int                 dm_valid; /* !=0 if successful diagnostics read */
} sfpBoard;

static LIST_HEAD(sfp_list);
static DEFINE_MUTEX(sfp_list_lock);

#define to_board_data(d)	container_of(d, sfpBoard, dev)

#define board_attr(name,fmt,arg...)				\
static ssize_t show_##name(struct device *_dev, struct device_attribute *attr, char *buf)	\
{								\
	sfpBoard *sfp = to_board_data(_dev);		\
	if(sfp) \
	return sprintf(buf, fmt, arg);				\
	else return sprintf(buf, "unable to find attrb\n");\
}								\
static DEVICE_ATTR(name, S_IRUGO, show_##name, NULL)

board_attr(id, "%02x\n", sfp->sfpMsaSerialId.id);
board_attr(extid, "%02x\n", sfp->sfpMsaSerialId.extId);
board_attr(connector, "%02x\n", sfp->sfpMsaSerialId.connector);
board_attr(txer, "%02x %02x %02x %02x %02x %02x %02x %02x\n",
	   sfp->sfpMsaSerialId.txer[0], sfp->sfpMsaSerialId.txer[1],
	   sfp->sfpMsaSerialId.txer[2], sfp->sfpMsaSerialId.txer[3],
	   sfp->sfpMsaSerialId.txer[4], sfp->sfpMsaSerialId.txer[5],
	   sfp->sfpMsaSerialId.txer[6], sfp->sfpMsaSerialId.txer[7]);
board_attr(encoding, "%02x\n", sfp->sfpMsaSerialId.encoding);
board_attr(br, "%02x\n", sfp->sfpMsaSerialId.br);
board_attr(length9u1, "%02x\n", sfp->sfpMsaSerialId.length9u1);
board_attr(length9u2, "%02x\n", sfp->sfpMsaSerialId.length9u2);
board_attr(length50u, "%02x\n", sfp->sfpMsaSerialId.length50u);
board_attr(length62dot5u, "%02x\n", sfp->sfpMsaSerialId.length62dot5u);
board_attr(lengthcopper, "%02x\n", sfp->sfpMsaSerialId.lengthCopper);
board_attr(vendorname, "%-16.16s\n", sfp->sfpMsaSerialId.vendorName);
board_attr(vendoroui, "%s\n", sfp->sfpMsaSerialId.vendorOui);
board_attr(vendorpn, "%-16.16s\n", sfp->sfpMsaSerialId.vendorPn);
board_attr(vendorrev, "%-4.4s\n", sfp->sfpMsaSerialId.vendorRev);
board_attr(vendorspecific, "%-32.32s\n",
	   sfp->sfpMsaSerialId.vendorSpecificData);
board_attr(wavelength, "%02x%02x\n", sfp->sfpMsaSerialId.wavelength[0],sfp->sfpMsaSerialId.wavelength[1]);
board_attr(ccbase, "%02x\n", sfp->sfpMsaSerialId.ccBase);
board_attr(options, "%02x %02x\n", sfp->sfpMsaSerialId.options[0],
	   sfp->sfpMsaSerialId.options[1]);
board_attr(minbr, "%02x\n", sfp->sfpMsaSerialId.minBr);
board_attr(maxbr, "%02x\n", sfp->sfpMsaSerialId.maxBr);
board_attr(vendorsn, "%-16.16s\n", sfp->sfpMsaSerialId.vendorSn);
board_attr(datecode, "%-8.8s\n", sfp->sfpMsaSerialId.dateCode);
board_attr(ccext, "%02x\n", sfp->sfpMsaSerialId.ccExt);
board_attr(dmtype, "%02x\n", sfp->sfpMsaSerialId.dmtype);

#if defined(ATI_PRODUCT_CONFIG)
board_attr(ativalidation, "%02x\n", sfp->sfpMsaSerialId.ativalidation);
#endif

board_attr(i2cbus, "%d\n", sfp->adap->nr);

static int atlat_sfp_read_sfp_diagnostics(sfpBoard * sfp);

#define board_diag_attr(name,fmt,arg...)				\
static ssize_t show_##name(struct device *_dev, struct device_attribute *attr, char *buf)	\
{								\
	sfpBoard *sfp = to_board_data(_dev);		\
	atlat_sfp_read_sfp_diagnostics(sfp);		\
	return sprintf(buf, fmt, arg);				\
}								\
static DEVICE_ATTR(name, S_IRUGO, show_##name, NULL)

// board attributes for SFP diagnostics monitoring
// Alarm and warning thresholds
board_attr(alarmthresholds, "%04x %04x %04x %04x %04x %04x %04x %04x %04x %04x\n",
	   sfp->sfpMsaDiagnostics.alarmThresholds[0],
	   sfp->sfpMsaDiagnostics.alarmThresholds[1],
	   sfp->sfpMsaDiagnostics.alarmThresholds[2],
	   sfp->sfpMsaDiagnostics.alarmThresholds[3],
	   sfp->sfpMsaDiagnostics.alarmThresholds[4],
	   sfp->sfpMsaDiagnostics.alarmThresholds[5],
	   sfp->sfpMsaDiagnostics.alarmThresholds[6],
	   sfp->sfpMsaDiagnostics.alarmThresholds[7],
	   sfp->sfpMsaDiagnostics.alarmThresholds[8],
	   sfp->sfpMsaDiagnostics.alarmThresholds[9]);
board_attr(warningthresholds, "%04x %04x %04x %04x %04x %04x %04x %04x %04x %04x\n",
	   sfp->sfpMsaDiagnostics.warningThresholds[0],
	   sfp->sfpMsaDiagnostics.warningThresholds[1],
	   sfp->sfpMsaDiagnostics.warningThresholds[2],
	   sfp->sfpMsaDiagnostics.warningThresholds[3],
	   sfp->sfpMsaDiagnostics.warningThresholds[4],
	   sfp->sfpMsaDiagnostics.warningThresholds[5],
	   sfp->sfpMsaDiagnostics.warningThresholds[6],
	   sfp->sfpMsaDiagnostics.warningThresholds[7],
	   sfp->sfpMsaDiagnostics.warningThresholds[8],
	   sfp->sfpMsaDiagnostics.warningThresholds[9]);
// NOTE: The following attributes are left because they will almost certainly be
// needed in a later implementation.
#if 0
	Calibration constants
board_attr(calrxpwr, "%08x %08x %08x %08x %08x\n",
	   sfp->sfpMsaDiagnostics.calRxPwr[0],
	   sfp->sfpMsaDiagnostics.calRxPwr[1],
	   sfp->sfpMsaDiagnostics.calRxPwr[2],
	   sfp->sfpMsaDiagnostics.calRxPwr[3],
	   sfp->sfpMsaDiagnostics.calRxPwr[4]);
board_attr(caltxlas, "%04x %04x\n",
	   sfp->sfpMsaDiagnostics.calTxLas[0],
	   sfp->sfpMsaDiagnostics.calTxLas[1]);
board_attr(caltxpwr, "%04x %04x\n",
	   sfp->sfpMsaDiagnostics.calTxPwr[0],
	   sfp->sfpMsaDiagnostics.calTxPwr[1]);
board_attr(caltemp, "%04x %04x\n",
	   sfp->sfpMsaDiagnostics.calTemp[0],
	   sfp->sfpMsaDiagnostics.calTemp[1]);
board_attr(calvolt, "%04x %04x\n",
	   sfp->sfpMsaDiagnostics.calVolt[0],
	   sfp->sfpMsaDiagnostics.calVolt[1]);
#endif
//	Diagnostic and control constants
board_diag_attr(temperature, "%04x\n",
	   sfp->sfpMsaDiagnostics.temperature);
board_diag_attr(vcc, "%04x\n",
	   sfp->sfpMsaDiagnostics.vcc);
board_diag_attr(txbias, "%04x\n",
	   sfp->sfpMsaDiagnostics.txBias);
board_diag_attr(txpower, "%04x\n",
	   sfp->sfpMsaDiagnostics.txPower);
board_diag_attr(rxpower, "%04x\n",
	   sfp->sfpMsaDiagnostics.rxPower);
board_diag_attr(statusandcontrol, "%02x\n",sfp->sfpMsaDiagnostics.statusAndControl);
// Alarm and warning flag bits
// SFF defines 4 separate bytes - implemented as 16 bit fields for convenience
board_diag_attr(alarmflags, "%04x\n",sfp->sfpMsaDiagnostics.alarmFlags);
board_diag_attr(warningflags, "%04x\n",sfp->sfpMsaDiagnostics.warningFlags);
// Extended module control/status - not in sys file
// User accessible EEPROM locations - not implemented

static sfpBoard* atlat_sfp_find_sfp_locked(int boardIndex, int portIndex)
{
	sfpBoard *ret_sfp = NULL;
	sfpBoard *sfp;

	list_for_each_entry(sfp, &sfp_list, list) {
		if (sfp->boardIndex == boardIndex &&
		    sfp->portIndex == portIndex) {
			ret_sfp = sfp;
			break;
		}
	}

	return ret_sfp;
}

static sfpBoard* atlat_sfp_find_sfp(int boardIndex, int portIndex)
{
	sfpBoard *sfp;

	mutex_lock(&sfp_list_lock);
	sfp = atlat_sfp_find_sfp_locked(boardIndex, portIndex);
	mutex_unlock(&sfp_list_lock);

	return sfp;
}

static int atlat_sfp_supports_diagnostics(sfpBoard * sfp)
{
	int ret = 0;

	ret = (sfp->sfpMsaSerialId.dmtype & SFP_DDM_SUPPORT_MASK);

	return ret;
}

static ssize_t show_eeprom(struct device *_dev, struct device_attribute *attr, char *buf)
{
	sfpBoard *sfp = to_board_data(_dev);
	ssize_t s = 0;
	ssize_t l;
	int i;
	unsigned char* p = (void*)&sfp->sfpMsaSerialId;

	for (i = 0; i < sizeof(sfp->sfpMsaSerialId); i++)
	{
		if (i % 16 == 0)
		{
			l = sprintf(buf, "%s%04x:", (i == 0 ? "" : "\n"), i);
			buf += l;
			s += l;
		}

		l = sprintf(buf, " %02x", *p);
		buf += l;
		s += l;

		p++;
	}

	l = sprintf(buf, " \n");
	buf += l;
	s += l;

	return s;
}
static DEVICE_ATTR(eeprom, S_IRUGO, show_eeprom, NULL);


static void atlat_sfp_device_release(struct device *dev)
{
	sfpBoard *sfp = to_board_data(dev);
	if(!sfp) return;

	dev_dbg(dev, "Release sfp\n");

	kfree(sfp);
}

static int atlat_sfp_add_board(sfpBoard * sfp)
{
	int err = 0;

	err |= device_create_file(&sfp->dev, &dev_attr_eeprom);
	err |= device_create_file(&sfp->dev, &dev_attr_id);
	err |= device_create_file(&sfp->dev, &dev_attr_vendorsn);
	err |= device_create_file(&sfp->dev, &dev_attr_datecode);
	err |= device_create_file(&sfp->dev, &dev_attr_vendorpn);
	err |= device_create_file(&sfp->dev, &dev_attr_vendorname);
	err |= device_create_file(&sfp->dev, &dev_attr_txer);
	err |= device_create_file(&sfp->dev, &dev_attr_length9u1);
	err |= device_create_file(&sfp->dev, &dev_attr_vendorspecific);

	err |= device_create_file(&sfp->dev, &dev_attr_extid);
	err |= device_create_file(&sfp->dev, &dev_attr_connector);
	err |= device_create_file(&sfp->dev, &dev_attr_encoding);
	err |= device_create_file(&sfp->dev, &dev_attr_br);
	err |= device_create_file(&sfp->dev, &dev_attr_length9u2);
	err |= device_create_file(&sfp->dev, &dev_attr_length50u);
	err |= device_create_file(&sfp->dev, &dev_attr_length62dot5u);
	err |= device_create_file(&sfp->dev, &dev_attr_lengthcopper);
	err |= device_create_file(&sfp->dev, &dev_attr_vendoroui);
	err |= device_create_file(&sfp->dev, &dev_attr_vendorrev);
	err |= device_create_file(&sfp->dev, &dev_attr_wavelength);
	err |= device_create_file(&sfp->dev, &dev_attr_ccbase);
	err |= device_create_file(&sfp->dev, &dev_attr_options);
	err |= device_create_file(&sfp->dev, &dev_attr_minbr);
	err |= device_create_file(&sfp->dev, &dev_attr_maxbr);
	err |= device_create_file(&sfp->dev, &dev_attr_ccext);
	err |= device_create_file(&sfp->dev, &dev_attr_dmtype);

#if defined(ATI_PRODUCT_CONFIG)
	err |= device_create_file(&sfp->dev, &dev_attr_ativalidation); // CR6970
#endif

	if (sfp->adap) {
		err |= device_create_file(&sfp->dev, &dev_attr_i2cbus);
	}
//      device_create_file(&sfp->dev, &dev_attr_iocontrol);
	if (err)
		dev_err(&sfp->dev, "Error adding device files\n");

	return err;
}

static int atlat_sfp_add_diagnostics(sfpBoard * sfp)
{
	int err = 0;

	err |= device_create_file(&sfp->dev, &dev_attr_alarmthresholds);
	err |= device_create_file(&sfp->dev, &dev_attr_warningthresholds);
	err |= device_create_file(&sfp->dev, &dev_attr_temperature);
	err |= device_create_file(&sfp->dev, &dev_attr_vcc);
	err |= device_create_file(&sfp->dev, &dev_attr_txbias);
	err |= device_create_file(&sfp->dev, &dev_attr_txpower);
	err |= device_create_file(&sfp->dev, &dev_attr_rxpower);
	err |= device_create_file(&sfp->dev, &dev_attr_statusandcontrol);
	err |= device_create_file(&sfp->dev, &dev_attr_alarmflags);
	err |= device_create_file(&sfp->dev, &dev_attr_warningflags);

	if (err)
		dev_err(&sfp->dev, "Error adding diagnostics device files\n");

	return err;
}

static int atlat_sfp_delete_board(sfpBoard * sfp)
{
	int err = 0;

	device_remove_file(&sfp->dev, &dev_attr_eeprom);
	device_remove_file(&sfp->dev, &dev_attr_id);
	device_remove_file(&sfp->dev, &dev_attr_extid);
	device_remove_file(&sfp->dev, &dev_attr_connector);
	device_remove_file(&sfp->dev, &dev_attr_txer);
	device_remove_file(&sfp->dev, &dev_attr_encoding);
	device_remove_file(&sfp->dev, &dev_attr_br);
	device_remove_file(&sfp->dev, &dev_attr_length9u1);
	device_remove_file(&sfp->dev, &dev_attr_length9u2);
	device_remove_file(&sfp->dev, &dev_attr_length50u);
	device_remove_file(&sfp->dev, &dev_attr_length62dot5u);
	device_remove_file(&sfp->dev, &dev_attr_lengthcopper);
	device_remove_file(&sfp->dev, &dev_attr_vendorname);
	device_remove_file(&sfp->dev, &dev_attr_vendoroui);
	device_remove_file(&sfp->dev, &dev_attr_vendorpn);
	device_remove_file(&sfp->dev, &dev_attr_vendorrev);
	device_remove_file(&sfp->dev, &dev_attr_vendorspecific);
	device_remove_file(&sfp->dev, &dev_attr_wavelength);
	device_remove_file(&sfp->dev, &dev_attr_ccbase);
	device_remove_file(&sfp->dev, &dev_attr_options);
	device_remove_file(&sfp->dev, &dev_attr_minbr);
	device_remove_file(&sfp->dev, &dev_attr_maxbr);
	device_remove_file(&sfp->dev, &dev_attr_vendorsn);
	device_remove_file(&sfp->dev, &dev_attr_datecode);
	device_remove_file(&sfp->dev, &dev_attr_ccext);
	device_remove_file(&sfp->dev, &dev_attr_dmtype);

#if defined(ATI_PRODUCT_CONFIG)
	device_remove_file(&sfp->dev, &dev_attr_ativalidation);// CR6970
#endif

	if (sfp->adap) {
		device_remove_file(&sfp->dev, &dev_attr_i2cbus);
	}
//      device_remove_file(&sfp->dev, &dev_attr_iocontrol);
	return err;
}

static int atlat_sfp_delete_diagnostics(sfpBoard * sfp)
{
	int err = 0;

	device_remove_file(&sfp->dev, &dev_attr_alarmthresholds);
	device_remove_file(&sfp->dev, &dev_attr_warningthresholds);
	device_remove_file(&sfp->dev, &dev_attr_temperature);
	device_remove_file(&sfp->dev, &dev_attr_vcc);
	device_remove_file(&sfp->dev, &dev_attr_txbias);
	device_remove_file(&sfp->dev, &dev_attr_txpower);
	device_remove_file(&sfp->dev, &dev_attr_rxpower);
	device_remove_file(&sfp->dev, &dev_attr_statusandcontrol);
	device_remove_file(&sfp->dev, &dev_attr_alarmflags);
	device_remove_file(&sfp->dev, &dev_attr_warningflags);

	return err;
}

static struct bus_type atlat_sfp_bustype = {
	.name = "sfp",
};


#if defined(ATI_PRODUCT_CONFIG)

#define SUPPORTED_VENDOR_NAME_SFP_TABLE_SIZE 2
// FYI: Each vendor name MUST match SFP_MSA_SERIAL_VENDOR_NAME_BYTES in length.
const char* SupportedVendorNameSfpTable[SUPPORTED_VENDOR_NAME_SFP_TABLE_SIZE] = 
{
   "ATI             ", 
   "Allied Telesis  ",
};

#define ATI_SFP_UNKNOWN                             0
#define ATI_SFP_KNOWN_COMPATIBLE_VENDOR             1
#define ATI_SFP_KNOWN_INCOMPATIBLE_VENDOR           2
#define ATI_SFP_KNOWN_COMPATIBLE_VENDOR_AND_MODEL   3
#define ATI_SFP_KNOWN_INCOMPATIBLE_VENDOR_AND_MODEL 4

#define SFP_STR_TABLE_SIZE 5
const char* SfpStrTable[SFP_STR_TABLE_SIZE] = 
{
  "Unknown",
  "Compatible Vendor, Unknown Model",
  "Incompatible Vendor",
  "Compatible Vendor & Model",
  "Incompatible Vendor & Model",
};

#define SUPPORTED_STD_SFP_TABLE_SIZE 6
// FYI: Each Standard SFP name MUST match SFP_MSA_SERIAL_VENDOR_PN_BYTES in length.
// This is a superset list IT MUST contain all of the suppported SFPs including
// those listed in the "SupportedHotSfpTable".
const char* SupportedStdSfpTable[SUPPORTED_STD_SFP_TABLE_SIZE] = 
{
   "AT-SPFXBD-LC-13 ",
   "AT-SPTX         ",
   "AT-SPBD10-13    ",
   "AT-SPBD20-13/I  ",
   "AT-TN-P015-A    ",
   "AT-SPBD20EPON-13", 
};

#define SUPPORTED_HOT_SFP_TABLE_SIZE 1
// FYI: Each HOT SFP name MUST match SFP_MSA_SERIAL_VENDOR_PN_BYTES in length.
const char* SupportedHotSfpTable[SUPPORTED_HOT_SFP_TABLE_SIZE] = 
{
   "AT-SPBD20EPON-13" 
};

static void ati_atlat_sfp_notify(sfpBoard * sfp, const int add)
{
  __u8 tempVendorName[SFP_MSA_SERIAL_VENDOR_NAME_BYTES+1]; // one extra byte for NULL char terminator
  __u8 tempVendorPn[SFP_MSA_SERIAL_VENDOR_PN_BYTES+1];     // one extra byte for NULL char terminator
  int i = 0;

  if (NULL == sfp) {return;}

  memset(&tempVendorName, 0, SFP_MSA_SERIAL_VENDOR_NAME_BYTES+1);
  memset(&tempVendorPn,   0, SFP_MSA_SERIAL_VENDOR_PN_BYTES+1);

  // Strip off any trailing spaces
  for (i = 0; i < SFP_MSA_SERIAL_VENDOR_NAME_BYTES; i++)
  {
	if (32 == sfp->sfpMsaSerialId.vendorName[i]) {break;}
	tempVendorName[i] = sfp->sfpMsaSerialId.vendorName[i];
  }

  // Strip off any trailing spaces
  for (i = 0; i < SFP_MSA_SERIAL_VENDOR_PN_BYTES; i++)
  {
	if (32 == sfp->sfpMsaSerialId.vendorPn[i]) {break;}
	tempVendorPn[i] = sfp->sfpMsaSerialId.vendorPn[i];
  }

  printk(KERN_CRIT "\nINFO: SFP Update %s - %s/%s/%s\n", (add)? "ADD" : "DEL", tempVendorName, tempVendorPn, SfpStrTable[sfp->sfpMsaSerialId.ativalidation]);
}

static void ati_atlat_sfp_device_validation(sfpBoard * sfp)
{
  int knownVendor   = ATI_SFP_UNKNOWN;
  int SupportedName = 0;
  int i             = 0;

  if (NULL == sfp) {return;}

  sfp->sfpMsaSerialId.ativalidation = ATI_SFP_KNOWN_INCOMPATIBLE_VENDOR_AND_MODEL;

  // Check the Vendor name Table for a match
  for (i=0; i < SUPPORTED_VENDOR_NAME_SFP_TABLE_SIZE; i++)
  {
	if (strncmp(SupportedVendorNameSfpTable[i], sfp->sfpMsaSerialId.vendorName, SFP_MSA_SERIAL_VENDOR_NAME_BYTES) == 0)
	{
	  pr_debug("DBG: %s@%d: Supported vendor found - %s\n", __func__, __LINE__, sfp->sfpMsaSerialId.vendorName);
	  knownVendor = ATI_SFP_KNOWN_COMPATIBLE_VENDOR;
	  break;
	}
  }

  // Check to see if we found a valid vendor name, if not exit!
  if (knownVendor != ATI_SFP_KNOWN_COMPATIBLE_VENDOR)
  {
	sfp->sfpMsaSerialId.ativalidation = ATI_SFP_KNOWN_INCOMPATIBLE_VENDOR;
	pr_debug("DBG: %s@%d: KIV - EXIT - audit=%s\n", __func__, __LINE__, SfpStrTable[sfp->sfpMsaSerialId.ativalidation]);
	return;
  }

  // Check the Superset Standard SFP Table for a match
  for (i=0; i < SUPPORTED_STD_SFP_TABLE_SIZE; i++)
  {
    if (strncmp(SupportedStdSfpTable[i], sfp->sfpMsaSerialId.vendorPn, SFP_MSA_SERIAL_VENDOR_PN_BYTES) == 0)
    {
	  SupportedName = 1;
      pr_debug("DBG: %s@%d: Supported name found - %s\n", __func__, __LINE__, sfp->sfpMsaSerialId.vendorPn);
    }
  }

  // See if we have an unsupported ATI device?
  if (0 == SupportedName)
  {
	sfp->sfpMsaSerialId.ativalidation = ATI_SFP_KNOWN_COMPATIBLE_VENDOR;
	pr_debug("DBG: %s@%d: KCV - EXIT - audit=%s\n", __func__, __LINE__, SfpStrTable[sfp->sfpMsaSerialId.ativalidation]);
	return;
  }

#if defined(ATI_PRODUCT_CONFIG) && defined(DMP_X_ALLIEDTELESIS_COM_SFPEXTENDEDTHERMAL_1)
  {
	int hotSfpFound = 0;

	// See if this SFP has been tagged as a High Thermal SFP (if its known to run "hot")
	for (i=0; i < SUPPORTED_HOT_SFP_TABLE_SIZE; i++)
	{
	  if (strncmp(SupportedHotSfpTable[i], sfp->sfpMsaSerialId.vendorPn, SFP_MSA_SERIAL_VENDOR_PN_BYTES) == 0)
	  {
		hotSfpFound = 1;
		pr_debug("DBG: %s@%d: High Thermal Support SFP found - %s\n", __func__, __LINE__, sfp->sfpMsaSerialId.vendorPn);
		break;
	  }
	}

	// See if we have found a "hot" sfp
    if (hotSfpFound) 
    {
	  // See if the UBOOT parm is present to allow a "hot" sfp
	  if (kerSysNvRamGetSfpHighThermalLoadSupport() > 0)
	  {
		// we are good, allow this sfp to run
		sfp->sfpMsaSerialId.ativalidation = ATI_SFP_KNOWN_COMPATIBLE_VENDOR_AND_MODEL;
		pr_debug("DBG: %s@%d: KCVM - EXIT - audit=%s\n", __func__, __LINE__, SfpStrTable[sfp->sfpMsaSerialId.ativalidation]);
		return;
	  }
	  else
	  {
		// do not allow this sfp to run
		sfp->sfpMsaSerialId.ativalidation = ATI_SFP_KNOWN_INCOMPATIBLE_VENDOR_AND_MODEL;
		pr_debug("DBG: %s@%d: KIVM - EXIT - audit=%s\n", __func__, __LINE__, SfpStrTable[sfp->sfpMsaSerialId.ativalidation]);
		return;
	  }
	}
  }
#endif

  // We have validated vendor and model!
  sfp->sfpMsaSerialId.ativalidation = ATI_SFP_KNOWN_COMPATIBLE_VENDOR_AND_MODEL;
  pr_debug("DBG: %s@%d: KCVM - EXIT - audit=%s\n", __func__, __LINE__, SfpStrTable[sfp->sfpMsaSerialId.ativalidation]);
  return;
}
#endif

static int atlat_sfp_device_register(sfpBoard * sfp)
{
	int ret;

	sfp->dev.release = atlat_sfp_device_release;
	sfp->dev.bus = &atlat_sfp_bustype;

	dev_set_name(&sfp->dev, "%d.%d", sfp->boardIndex, sfp->portIndex);

	ret = device_register(&sfp->dev);
	dev_dbg(&sfp->dev, "Register\n");

	return ret;
}

static void atlat_sfp_device_unregister(sfpBoard * sfp)
{
	dev_dbg(&sfp->dev, "Unregister\n");
	device_unregister(&sfp->dev);

	return;
}

static int atlat_sfp_load_sfp_diagnostics(sfpBoard * sfp, u8 * data)
{
	int i;
	int j;
	int alarmIndex = 0;
	int warnIndex = 0;
	__u8 checkSum = 0;

	/* Calculate checksum and zero data if incorrect */
	for (i = 0; i <= 95; i++ ) {
		checkSum = (__u8)data[i];
	}

	if (checkSum == data[95]) {
		/*	Alarm thresholds and warning thresholds are split out into two
			separate arrays at this point for later processing convenience.
			Thresholds are stored on the device in an alternating alarm high/low,
			warning high/low order e.g:
			Temperature Alarm High - 2 bytes
			Temperature Alarm Low - 2 bytes
			Temperature Warn High - 2 bytes
			Temperature Warn Low - 2 bytes
			Voltage Alarm High - 2 bytes etc. */
		for (i = 0; i < (SFP_MSA_DIAG_THRESHOLD_FIELDS); i++) {
			j = (i * 2);

			if ((i % 4) < 2) {
				sfp->sfpMsaDiagnostics.alarmThresholds[alarmIndex] =
					((data[j] << 8) + data[j+1]);
				alarmIndex++;
			} else	{
				sfp->sfpMsaDiagnostics.warningThresholds[warnIndex] =
					((data[j] << 8) + data[j+1]);
				warnIndex++;
			}
		}
		sfp->sfpMsaDiagnostics.temperature = ((data[96] << 8) + data[97]);
		sfp->sfpMsaDiagnostics.vcc = ((data[98] << 8) + data[99]);
		sfp->sfpMsaDiagnostics.txBias = ((data[100] << 8) + data[101]);
		sfp->sfpMsaDiagnostics.txPower = ((data[102] << 8) + data[103]);
		sfp->sfpMsaDiagnostics.rxPower = ((data[104] << 8) + data[105]);
		sfp->sfpMsaDiagnostics.statusAndControl = data[110];
		/* Alarm and warning flag bytes become words - lowest byte has MSB */
		sfp->sfpMsaDiagnostics.alarmFlags = ((data[112] << 8) + data[113]);
		sfp->sfpMsaDiagnostics.warningFlags = ((data[116] << 8) + data[117]);
	} else {
		memset((u8 *) & sfp->sfpMsaDiagnostics, 0, sizeof(sfpMsaDiagnosticsT));
		pr_err("%s:%u SFP diagnostics checksum failed\n", __func__,
			__LINE__);
		return -1;
	}
	return 0;
}

static int atlat_sfp_read_sfp_diagnostics(sfpBoard * sfp)
{
	int i;
	__u8 txData = 0;
	struct timeval tv,tv2;
	__u8  data[256];

	if (sfp == NULL || !sfp->valid) {
		pr_err("%s:%u corrupt sfp structure sfp %p\n",
		       __func__, __LINE__, sfp);
		return -1;
	}

	/*	SFP's that don't have an attached I2c adaptor will have their I2C code
		accessed via user space, so this OpenHPI triggered I2C read cannot
		take place. However, it is not an error so we return zero in this case */

	if (sfp->adap == NULL) {
		return 0;
	}

	if (!atlat_sfp_supports_diagnostics(sfp)) {
		return 0;
	}

	/* minimum measurement cycle: 1.75 seconds */
	if (time_after(jiffies, sfp->dm_last_updated + (HZ * 7 / 4))
	    || !sfp->dm_valid) {
		pr_debug("%s:%u I2C read adapter %p %d addr %x\n", __func__, __LINE__,
			sfp->adap, sfp->adap->nr, SFP_DIAG_I2C_ADDRESS);
		do_gettimeofday(&tv);

		for (i = 0; i < SFP_DIAG_RETRIES; i++) {
			if (0 !=
				atlat_i2c_transaction_retry(sfp->adap, SFP_DIAG_I2C_ADDRESS,
							sizeof(sfpMsaDiagnosticsT), 1,
							(u8 *)data,
							&txData)) {
				msleep_interruptible(5);
				do_gettimeofday(&tv2);
				if ((tv2.tv_sec - tv.tv_sec) > 10) {
					printk(KERN_EMERG
						"SFP in port %d.%d is locking up the I2C bus. Please remove.\n",
						sfp->boardIndex, sfp->portIndex);
					do_gettimeofday(&tv);
				}
				continue;
			}
			break;
		}

		pr_debug("%s:%u I2C read adapter %p %d addr %x i %d\n", __func__,
			__LINE__, sfp->adap, sfp->adap->nr, SFP_DIAG_I2C_ADDRESS, i);
		if (i >= SFP_DIAG_RETRIES) {
			pr_err("%s:%u I2C bad transfer\n", __func__,
				__LINE__);
			return 0;
		}

		sfp->dm_last_updated = jiffies;
		sfp->dm_valid = 1;

		return (atlat_sfp_load_sfp_diagnostics(sfp, data));
	}
	return 0;
}

static int atlat_sfp_get_data(sfpBoard *sfp)
{
	struct i2c_adapter *adap = sfp->adap;
	struct timeval tv, tv2;
#if !defined(ATI_PRODUCT_CONFIG)
	int i, j;
#else
    int i;
#endif
#if !defined(ATI_PRODUCT_CONFIG)
	__u32 txer;
#endif
	__u8 txData = 0;
#if !defined(ATI_PRODUCT_CONFIG)
	__u32 invalid;
#endif

	pr_debug("%s:%u I2C read adapter %p %d addr %x\n", __func__, __LINE__,
	    adap, adap->nr, SFP_I2C_ADDRESS);
	//if we don't do this sleep we sometimes get successful reads full of garbage.
	msleep_interruptible(10);
	do_gettimeofday(&tv);
	for (i = 0; i < SFP_STARTUP_RETRIES; i++) {
		if (0 !=
		    atlat_i2c_transaction_retry(adap, SFP_I2C_ADDRESS,
						sizeof(sfpMsaSerialIdT), 1,
						(u8 *) & sfp->sfpMsaSerialId,
						&txData)) {
			msleep_interruptible(5);
			do_gettimeofday(&tv2);
			pr_debug("%s: SFP %d.%d retry %d time %ld : %ld\n",
			    __func__, sfp->boardIndex, sfp->portIndex,
			    i, tv2.tv_sec, tv2.tv_usec);
			if ((tv2.tv_sec - tv.tv_sec) > 10) {
				printk(KERN_EMERG
				       "SFP in port %d.%d is locking up the I2C bus. Please remove.\n",
				       sfp->boardIndex, sfp->portIndex);
				do_gettimeofday(&tv);
				i = SFP_STARTUP_RETRIES;
				break;
			}
			memset(&sfp->sfpMsaSerialId, 0, sizeof(sfpMsaSerialIdT));
			continue;
		}
/*
 * ATI_PRODUCT_CONFIG: See CR 5921. Removing the sanity check because they don't 
 * necessarily guarantee that SFP EEPROM content is invalid. 
 */
#if !defined(ATI_PRODUCT_CONFIG)
/*		CR24623 Sanity check the SFP erprom data.
 *		For one particular type of SFP, it was found that the i2c transaction above 
 *		would succeed, but the serial ID data returned was all zeroes. If the transaction
 *		was retried after a short delay, valid serial ID data was read. It seems as though
 *		this type of SFP took longer to startup than others.
 *		Note: this issue only occurred when the SFP was present at startup on x900 units,
 *		and not when the SFP was hot inserted after bootup.
 *		For the sanity check, we'll use the transceiver field.
 *		At least 1 bit of the transceiver field MUST be set.
 *		(Ref pg34 "Small Form-factor Pluggable (SFP) Transceiver Multisource Agreement (MSA)" Sept 14 2000)
 */
		txer = 0;
		for (j = 0; j < SFP_MSA_SERIAL_TXER_BYTES; j++) {
			txer += sfp->sfpMsaSerialId.txer[j];
		}
		if (txer == 0) {
			pr_debug("%s: SFP (%d.%d) serial eeprom data invalid. retrying...\n",
			    __func__, sfp->boardIndex, sfp->portIndex);
			msleep_interruptible(5);
			memset((u8 *)&sfp->sfpMsaSerialId, 0, sizeof(sfpMsaSerialIdT));
			continue;
		}

		invalid = 0;
		for (j = 0; j < SFP_MSA_SERIAL_VENDOR_PN_BYTES; j++) {
			if (sfp->sfpMsaSerialId.vendorPn[j] == 0)
				break;
			if (sfp->sfpMsaSerialId.vendorPn[j] == 0xff) {
				invalid = 1;
			}
		}

		if (!invalid) {
			for (j = 0; j < SFP_MSA_SERIAL_VENDOR_SPECIFIC_BYTES; j++) {
				if (sfp->sfpMsaSerialId.vendorSpecificData[j] == 0)
					break;
				if (sfp->sfpMsaSerialId.vendorSpecificData[j] == 0xff) {
					invalid = 1;
				}
			}
		}

		if (invalid) {
			pr_debug("%s: SFP %d.%d invalid %d\n", __func__,
			    sfp->boardIndex, sfp->portIndex, invalid);
			pr_debug("%s: SFP (%d.%d) serial eeprom data invalid. retrying...\n",
			    __func__, sfp->boardIndex, sfp->portIndex);
			msleep_interruptible(5);
			memset((u8 *)&sfp->sfpMsaSerialId, 0, sizeof(sfpMsaSerialIdT));
			continue;
		}
#endif
		break;
	}

	pr_debug("%s:%u I2C read adapter %p %d addr %x i %d\n", __func__,
	    __LINE__, adap, adap->nr, SFP_I2C_ADDRESS, i);

	return (i < SFP_STARTUP_RETRIES) ? 0 : -1;
}

#if !defined(ATI_PRODUCT_CONFIG)
static ssize_t store_sfp_diag_bin(struct file *file, struct kobject *kobj, struct bin_attribute *attr,
                             char *buf, loff_t off, size_t count)
#else
static ssize_t store_sfp_diag_bin(struct kobject *kobj, struct bin_attribute *attr,
                             char *buf, loff_t off, size_t count)
#endif
{
	sfpXfpPluggableData pluggableDiagnosticsData;
	sfpBoard *sfp = to_board_data(container_of(kobj, struct device, kobj));

	if (count < sizeof(pluggableDiagnosticsData)) {
		printk(KERN_ERR "%s:%u incorrect size for sfp diagnostics data structure sfp %p\n",
		       __FUNCTION__, __LINE__, buf);
		return count;
	}

	memcpy(&pluggableDiagnosticsData, (u8 *)buf, count);

	// An additional check on the board and port index passed
	if (sfp && atlat_sfp_present(pluggableDiagnosticsData.boardIndex,
								 pluggableDiagnosticsData.port)) {
		atlat_sfp_load_sfp_diagnostics(sfp, pluggableDiagnosticsData.data);
	} else {
		printk(KERN_ERR "%s:%u Cannot find SFP at board index %u and port index %u\n",
		       __func__, __LINE__,
               pluggableDiagnosticsData.boardIndex, pluggableDiagnosticsData.port);
	}
	return count;
}

static struct bin_attribute sfp_diag_bin_attr = {
	.attr = {
	.name = "sfp_diag_data",
	.mode = S_IWUGO,
	},
	.size = sizeof(sfpXfpPluggableData),
	.write = store_sfp_diag_bin,
};

static int atlat_sfp_add_sfp_common(struct i2c_adapter *adap,
                         int boardIndex,
                         int portIndex,
                         const u8 *data)
{
	sfpBoard *sfp;
	int ret;

	if ((adap && data) || (!adap && !data)) {
		pr_err("%s:%u Need to supply exactly one of adap and data\n",
		       __func__, __LINE__);
		return -EINVAL;
	}

	mutex_lock(&sfp_list_lock);
	/* check if an entry already exists for this SFP */
	if (atlat_sfp_find_sfp_locked(boardIndex, portIndex)) {
		pr_debug("%s: SFP %d.%d already exists\n", __func__, boardIndex, portIndex);
		mutex_unlock(&sfp_list_lock);
		return 0;
	}

	sfp = kzalloc(sizeof(sfpBoard), GFP_KERNEL);
	if (sfp == NULL) {
		pr_err("%s:%u bad memory allocation\n", __func__, __LINE__);
		mutex_unlock(&sfp_list_lock);
		return -ENOMEM;
	}
	sfp->boardIndex = boardIndex;
	sfp->portIndex = portIndex;

	if (adap) {
		sfp->adap = adap;
		ret = atlat_sfp_get_data(sfp);
		if (ret < 0) {
			pr_err("%s:%u I2C bad transfer\n", __func__, __LINE__);
			kfree(sfp);
			mutex_unlock(&sfp_list_lock);
			return -EIO;
		}
	} else if (data) {
		memcpy(&sfp->sfpMsaSerialId, data, sizeof(sfpMsaSerialIdT));
	}

	pr_debug("%s: SFP %d.%d\n", __func__, boardIndex, portIndex);
	pr_debug("%s:   id %d\n", __func__, sfp->sfpMsaSerialId.id);
	pr_debug("%s:   vendorName '%.16s'\n", __func__,
	    sfp->sfpMsaSerialId.vendorName);
	pr_debug("%s:   vendorOui '%.3s'\n", __func__,
	    sfp->sfpMsaSerialId.vendorOui);
	pr_debug("%s:   vendorPn '%.16s'\n", __func__,
	    sfp->sfpMsaSerialId.vendorPn);
	pr_debug("%s:   vendorRev '%.4s'\n", __func__,
	    sfp->sfpMsaSerialId.vendorRev);

#if defined(ATI_PRODUCT_CONFIG)
	ati_atlat_sfp_device_validation(sfp);
	ati_atlat_sfp_notify(sfp, 1);
#endif

	if ((ret = atlat_sfp_device_register(sfp)) != 0) {
		pr_debug("%s:%u atlat_sfp_device_register failed %x\n", __func__,
		    __LINE__, (unsigned int)ret);
		kfree(sfp);
		mutex_unlock(&sfp_list_lock);
		return -ENODEV;
	}
	atlat_sfp_add_board(sfp);

	sfp->valid = 1;
	list_add_tail(&sfp->list, &sfp_list);

	// Add sysf diagnostics support for those sfp's which support it
	if (atlat_sfp_supports_diagnostics(sfp)) {
		if (data) {
			//add file in support of user space I2C read
			ret = device_create_bin_file(&sfp->dev, &sfp_diag_bin_attr);
			if (ret)
				dev_err(&sfp->dev, "Failed to create SFP diagnostics bin file.\n");
		}
		atlat_sfp_add_diagnostics(sfp);
	}
	mutex_unlock(&sfp_list_lock);

#if defined(ATI_PRODUCT_CONFIG)
	pr_debug("%s: ADD SFP - ASSERT MSG_NETLINK_BRCM_ATI_SFP_STATUS_UPDATE\n", __func__);
	kerSysSendtoMonitorTask(MSG_NETLINK_BRCM_ATI_SFP_STATUS_UPDATE, NULL, 0);
#endif

	return 0;
}

int atlat_sfp_add_sfp(struct i2c_adapter *adap, int boardIndex, int portIndex)
{
	return atlat_sfp_add_sfp_common(adap, boardIndex, portIndex, NULL);
}

int atlat_sfp_add_sfp_with_data(int boardIndex, int portIndex, const u8 *data)
{
	return atlat_sfp_add_sfp_common(NULL, boardIndex, portIndex, data);
}
EXPORT_SYMBOL(atlat_sfp_add_sfp_with_data);

int atlat_sfp_add_sfp_diagnostics_data(int boardIndex, int portIndex, u8 * data)
{
	sfpBoard *sfp = NULL;
	int ret = -1;

	mutex_lock(&sfp_list_lock);
	sfp = atlat_sfp_find_sfp_locked(boardIndex, portIndex);
	if (sfp) {
		ret = atlat_sfp_load_sfp_diagnostics(sfp, data);
	} else {
		pr_err("%s:%u Cannot find SFP\n", __func__, __LINE__);
	}
	mutex_unlock(&sfp_list_lock);

	return ret;
}
EXPORT_SYMBOL(atlat_sfp_add_sfp_diagnostics_data);

//SFP should have been deleted from the list when this is called.
static int atlat_sfp_do_delete_sfp(sfpBoard * sfp)
{
	if (sfp == NULL || !sfp->valid) {
		pr_err("%s:%u corrupt sfp structure sfp %p\n",
		       __func__, __LINE__, sfp);
		return -1;
	}

	sfp->valid = 0;
	atlat_sfp_delete_board(sfp);

	// Remove sysf diagnostics files for those sfp's which support it
	if (atlat_sfp_supports_diagnostics(sfp)) {
		atlat_sfp_delete_diagnostics(sfp);
		device_remove_bin_file(&sfp->dev, &sfp_diag_bin_attr);
	}
	atlat_sfp_device_unregister(sfp);
	/* the remainder of the cleanup occurs in atlat_sfp_device_release as
	   part of the device_unregister */

	return 0;
}

int atlat_sfp_delete_sfp(int boardIndex, int portIndex)
{
	sfpBoard *sfp;
	int ret = 0;

	mutex_lock(&sfp_list_lock);
	sfp = atlat_sfp_find_sfp_locked(boardIndex, portIndex);
	if (sfp) {
#if defined(ATI_PRODUCT_CONFIG)
	    ati_atlat_sfp_notify(sfp, 0);
 #endif
		list_del(&sfp->list);
		ret = atlat_sfp_do_delete_sfp(sfp);
	}
	mutex_unlock(&sfp_list_lock);

#if defined(ATI_PRODUCT_CONFIG)
	pr_debug("%s: DEL SFP - ASSERT MSG_NETLINK_BRCM_ATI_SFP_STATUS_UPDATE\n", __func__);
	kerSysSendtoMonitorTask(MSG_NETLINK_BRCM_ATI_SFP_STATUS_UPDATE, NULL, 0);
#endif

	return ret;
}

void atlat_sfp_delete_board_sfps(int boardIndex)
{
	sfpBoard *sfp;
	sfpBoard *n;

	mutex_lock(&sfp_list_lock);
	list_for_each_entry_safe(sfp, n, &sfp_list, list) {
		if (sfp->boardIndex == boardIndex) {
			list_del(&sfp->list);
			atlat_sfp_do_delete_sfp(sfp);
		}
	}
	mutex_unlock(&sfp_list_lock);
}

int atlat_sfp_present(int boardIndex, int portIndex)
{
	if (atlat_sfp_find_sfp(boardIndex, portIndex))
		return true;
	else
		return false;
}
EXPORT_SYMBOL(atlat_sfp_present);

#if !defined(ATI_PRODUCT_CONFIG)
static ssize_t store_sfp_bin(struct file *file, struct kobject *kobj, struct bin_attribute *attr,
                             char *buf, loff_t off, size_t count)
#else
static ssize_t store_sfp_bin(struct kobject *kobj, struct bin_attribute *attr,
                             char *buf, loff_t off, size_t count)
#endif
{
	sfpXfpPluggableData newPluggableData;

	if (count < sizeof (sfpXfpPluggableData)) {
		printk(KERN_ERR "%s:%u incorrect size for sfp data structure sfp %p\n",
		       __func__, __LINE__, buf);
		return count;
	}

	memcpy (&newPluggableData, (u8 *)buf, count);

	if (!atlat_sfp_present (newPluggableData.boardIndex, newPluggableData.port)) {
		atlat_sfp_add_sfp_with_data (newPluggableData.boardIndex, newPluggableData.port, newPluggableData.data);
	}

	return count;
}

static struct bin_attribute sfp_creation_bin_attr = {
	.attr = {
	.name = "new_sfp",
	.mode = S_IWUGO,
	},
	.size = sizeof(sfpXfpPluggableData),
	.write = store_sfp_bin,
};

int atlat_sfp_init(void)
{
	int ret;

	ret = bus_register(&atlat_sfp_bustype);

	if (!ret) {
		ret = bus_create_bin_file (&atlat_sfp_bustype, &sfp_creation_bin_attr);
	}

	return ret;
}

void atlat_sfp_exit(void)
{
	return;
}

//EXPORT_SYMBOL(atlat_sfp_init);
