/* ATI Persistent logging.
 * (C) 2011 Allied Telesis Labs.
 *
 * This code is licenced under the GPL.
 */

#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <ati_hwdefs.h>
#include <linux/crc32.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/nbuff.h>

#define VERSION     "1.0"
#define VER_STR     "v" VERSION " " __DATE__ " " __TIME__
#define PROC_ENTRY_NAME "atilog"
#define PROC_CTRL_ENTRY_NAME "atilogcmd"

extern tAtiPersistentLogMemoryMap *atiPersLogMemMap;
extern tAtiReservedMemoryMap *atiResMemMap;
extern unsigned atiPersistentLogSize;

// Find the nth string in the persistent logging structure.
//   After the first iteration... Should always return next string.  
//
static char *findnstr(void **laststr, loff_t n)
{
  loff_t cnt;
  unsigned index = (unsigned)(*laststr); 
  if (index) //ignore n... Just get next since this is a circular buffer
  {
    n = 1;
  }
  else
  {
    index=atiResMemMap->persistentLogDB.start;
  }
  if (index == atiResMemMap->persistentLogDB.end)
  {
    return(NULL);
  }
  cnt = 0;
  while (cnt!=n)
  {
    while (index != atiResMemMap->persistentLogDB.end && atiPersLogMemMap->persistentLog[index])
    {
      index = (index + 1) & (atiPersistentLogSize-1);
    }
    index = (index + 1) & (atiPersistentLogSize-1);
    if (index == atiResMemMap->persistentLogDB.end)
      return (NULL);
    cnt++;
  }
  *laststr = (void *)index;
  return (&(atiPersLogMemMap->persistentLog[index]));
}

// Initialize the seq state machine.
// 
static void *atilog_seq_start(struct seq_file *s, loff_t *pos)
{
  char *str;
  // pos is relative to start
//  seq_printf(s, "%s:%Ld:%p:%p\n", __FUNCTION__, *pos, s, s->private);
  str = findnstr(&(s->private), *pos);
  return(str);
}

// Don't look for the next entry, just return NULL so the seq state machine will display the data now.
static void *atilog_seq_next(struct seq_file *s, void *v, loff_t *pos)
{
  return NULL;
}

static void atilog_seq_stop(struct seq_file *sfile, void *v)
{
}

// Print the string supplied by seq_start.
static int atilog_seq_show(struct seq_file *s, void *v)
{
	char *str = (char *) v;
  seq_printf(s, "%s\n", str);
  return 0;
}

static struct seq_operations atilog_seq_ops =
{
  .start = atilog_seq_start,
  .next  = atilog_seq_next,
  .stop  = atilog_seq_stop,
  .show  = atilog_seq_show
};

static int atilog_proc_open(struct inode *inode, struct file *file)
{
  return seq_open(file, &atilog_seq_ops);
}

static ssize_t atilog_proc_write(struct file *s, const char *buf, size_t cnt, loff_t *pos)
{
  char *rtn;
  unsigned start = atiResMemMap->persistentLogDB.start;
  unsigned end = atiResMemMap->persistentLogDB.end;
  unsigned localCnt;

  rtn = strrchr(buf, '\n');
  if (rtn && (rtn-buf) == (cnt-1)) // found '\n' and is at the end of the string 
  {
    *rtn = '\0';
  }
  else
    cnt++;
  for(localCnt = 0; localCnt < cnt; localCnt++)
  {
    atiPersLogMemMap->persistentLog[end] = *buf;
    end = (end + 1) & (atiPersistentLogSize-1);
    if (end == start) 
    {
      // Find the new start...
      do
      {
        start = (start+1) & (atiPersistentLogSize-1);
      } while (!atiPersLogMemMap->persistentLog[start]);
      start = (start+1) & (atiPersistentLogSize-1);
    }
    buf++;
  }
  atiPersLogMemMap->persistentLog[end] = '\0'; // Double terminate for start/next/show

  // flush the persistent log cache
  cache_flush_len(atiPersLogMemMap->persistentLog, atiPersistentLogSize);
  atiResMemMap->persistentLogDB.start = start;
  atiResMemMap->persistentLogDB.end = end;
  atiResMemMap->persistentLogDB.CRC = crc32(0, &atiResMemMap->persistentLogDB, sizeof(atiResMemMap->persistentLogDB)-4);
  // flush the databasetable.
  cache_flush_len(&atiResMemMap->persistentLogDB, sizeof(atiResMemMap->persistentLogDB));

  if (!rtn) 
  {
    cnt--;
  }
  return(cnt);
}

static struct file_operations atilog_proc_ops =
{
  .owner   = THIS_MODULE,
  .open    = atilog_proc_open,
  .read    = seq_read,
  .write   = atilog_proc_write,
  .llseek  = seq_lseek,
  .release = seq_release
};

enum { NONE
  , CLEAR_LOG
};
static int proc_set_command(struct file *f, const char *buf, unsigned long cnt, void *data)
{

  char input[32];
  int cmd;

  if (cnt > 32)
      cnt = 32;

  if (copy_from_user(input, buf, cnt) != 0)
      return -EFAULT;
  input[cnt] = '\0';
  cmd = simple_strtoul(input, NULL, 10);

  switch (cmd)
  {
  case CLEAR_LOG:
    memset(&atiResMemMap->persistentLogDB, 0, sizeof(atiResMemMap->persistentLogDB));
    break;
  default:
    printk("INVALID COMMAND: cnt=%lu, command=%s, cmd=%d\n", cnt, input, cmd);
    break;
  }
  return cnt;
}

static int __init atiPersistentLog_init( void )
{
  struct proc_dir_entry *p;
  unsigned crc;

  if (atiResMemMap && atiPersLogMemMap)
  {
    // Verify the persistent ram space.
    crc = crc32(0, &atiResMemMap->persistentLogDB, sizeof(atiResMemMap->persistentLogDB)-4);
  
    // Create ther proc file system entry.
    p = create_proc_entry(PROC_ENTRY_NAME, 0, 0);
    if (!p) {
      printk("ATI Persistent Log: Failed to create proc file.\n");
      return 1;
    }
    p->proc_fops = &atilog_proc_ops;
  
    p = create_proc_entry(PROC_CTRL_ENTRY_NAME, 0, 0);
    if (!p) {
      printk("ATI Persistent Log: Failed to create ctrl proc file.\n");
      return 1;
    }
    p->read_proc=NULL;
    p->write_proc=proc_set_command;

    if (crc != atiResMemMap->persistentLogDB.CRC)
    {
      memset(&atiResMemMap->persistentLogDB, 0, sizeof(atiResMemMap->persistentLogDB));
    }
    else
      atiResMemMap->persistentLogDB.currStart = atiResMemMap->persistentLogDB.end;

    // Banner
    printk("ATI Persistent Log@%p:%dK(%s):%d:%d: %s\n", 
           atiPersLogMemMap, 
           atiPersistentLogSize/1024, 
           (atiResMemMap->persistentLogDB.CRC==crc)?"Valid":"Invalid", 
           atiResMemMap->persistentLogDB.start, 
           atiResMemMap->persistentLogDB.end, 
           VER_STR);
  }
  return 0;
}

late_initcall( atiPersistentLog_init );

