#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/rtnetlink.h>
#include <linux/netlink.h>
#include <net/sock.h>
#include <linux/in.h>
#include <linux/if.h>

//#include <../net/bridge/br_loopprot.h>
#include "br_loopprot.h"

static struct sock *loopprot_nl_sk = NULL;
static int loopprot_pid = 0;


typedef struct loopprot_msg_hdr 
{
    __u16 type;
    __u16 len;
} t_LOOPPROT_MSG_HDR;

typedef enum loopprot_msgtype 
{
    LOOPPROT_MSG_BASE = 0,
    LOOPPROT_MSG_REGISTER,  /* usr - > krnl -> usr */
    LOOPPROT_MSG_PKT,       /* krnl -> usr */
    LOOPPROT_MSG_LDF_SEND,  /* usr - > krnl */
    LOOPPROT_MSG_CONFIG,    /* usr - > krnl */
    LOOPPROT_MSG_MAX
} t_LOOPPROT_MSGTYPES;

typedef enum loopprot_ret_codes 
{
    LOOPPROT_SUCCESS = 0,
    LOOPPROT_GEN_ERR = 1,
} t_LOOPPROT_RET_CODE;

typedef struct loopprot_register 
{
    int code;
} t_LOOPPROT_REGISTER;

typedef struct loopprot_config 
{
    int enable;
} t_LOOPPROT_CONFIG;

typedef struct loopprot_pkt_info
{
    char                      br_name[IFNAMSIZ+1];
    char                      port_name[IFNAMSIZ+1];
    int                       port_no;
    int                       if_index;
    int                       data_len;
    __u32                     corr_tag;
    __u16                     tci;/* vlan id */
} t_LOOPPROT_PKT_INFO;


//static void loopprot_dump_buf(char *buf, int len)
//{
//    int i;
//    printk("========================KRNEL loopprot_process_skb START===================================\n");
//    for(i =0; i < len; i++)
//    {
//     printk("%02x", (unsigned char)buf[i]);
//     if(!((i+1)%2))
//         printk(" ");
//     if(!((i+1)%16))
//       printk("\n");
//    }
//    printk("\n");
//    printk("=======================KRNL loopprot_process_skb END====================================\n");
//}

//int loopprot_process_skb(struct net_bridge *br, struct sk_buff *skb)
int loopprot_process_skb(struct net_device *dev, struct sk_buff *skb)
{
    struct nlmsghdr     *new_nlh;
    char                *new_ptr   = NULL;
    struct sk_buff      *new_skb;
    t_LOOPPROT_PKT_INFO *pkt_info;
    int                  buf_size;
    int                  port_number;
    int                  ret;
    char                *br_name;
    int                  if_index;

    port_number = 0;
    br_name     = dev->name;
    if_index    = dev->ifindex;

    if (!loopprot_pid || !br_loopprot_enabled_get())
        return 0;

    buf_size = NLMSG_SPACE(skb->len + sizeof(t_LOOPPROT_MSG_HDR) + sizeof(t_LOOPPROT_PKT_INFO));
    new_skb  = alloc_skb(NLMSG_SPACE(buf_size), GFP_ATOMIC);

    if(!new_skb) {
        printk("loopprot_process_skb.c:%d %s() errr no mem\n", __LINE__, __FUNCTION__);
        return 0;
    }

    // init nlh header
    new_nlh              = (struct nlmsghdr *)new_skb->data;
    new_ptr              = NLMSG_DATA(new_nlh);
    new_nlh->nlmsg_len   = NLMSG_SPACE(buf_size);
    new_nlh->nlmsg_pid   = 0;
    new_nlh->nlmsg_flags = 0;
    skb_put(new_skb, NLMSG_SPACE(buf_size));
    ((t_LOOPPROT_MSG_HDR *)new_ptr)->type = LOOPPROT_MSG_PKT; //  send to usr app
    ((t_LOOPPROT_MSG_HDR *)new_ptr)->len  = sizeof(t_LOOPPROT_PKT_INFO);
//    new_skb->protocol                     = NETLINK_LOOPPROT;

    // skip past t_LOOPPROT_MSG_HDR
    new_ptr += sizeof(t_LOOPPROT_MSG_HDR);

    pkt_info = (t_LOOPPROT_PKT_INFO *)new_ptr;
    memset(pkt_info, 0, sizeof(t_LOOPPROT_PKT_INFO));

    if (br_name)
    {
       strncpy(pkt_info->br_name, br_name, IFNAMSIZ);
    }

    strncpy(pkt_info->port_name, br_name, IFNAMSIZ);
    pkt_info->port_no  = port_number;
    pkt_info->if_index = if_index;
//    pkt_info->data_len = skb->len-4;

    pkt_info->data_len = skb->len;
    pkt_info->corr_tag = (__u32)skb;

#if defined(CONFIG_BCM_VLAN) || defined(CONFIG_BCM_VLAN_MODULE)
    if(skb->vlan_count)
        pkt_info->tci = (skb->vlan_header[0] >> 16);
#endif /*  defined(CONFIG_BCM_VLAN) || defined(CONFIG_BCM_VLAN_MODULE) */
    pkt_info->tci = skb->vlan_tci; // ATI_BSP_PERSONALITY
    new_ptr += sizeof(t_LOOPPROT_PKT_INFO);

    memcpy(new_ptr, skb->data, skb->len);

    NETLINK_CB(new_skb).dst_group = 0;
    NETLINK_CB(new_skb).pid       = loopprot_pid;
//    loopprot_dump_buf((char *)new_ptr, skb->len);

    ret = netlink_unicast(loopprot_nl_sk, new_skb, loopprot_pid, MSG_DONTWAIT);

    return 1;
} /* loopprot_process_skb */
EXPORT_SYMBOL(loopprot_process_skb);

static void loopprot_nl_process_registration(struct sk_buff *skb)
{
    struct nlmsghdr *nlh     = (struct nlmsghdr *)skb->data;
    struct sk_buff *new_skb  = NULL;
    char  *new_ptr           = NULL;
    struct nlmsghdr *new_nlh = NULL;
    int buf_size;
    int ret;

    loopprot_pid = nlh->nlmsg_pid;

    buf_size = NLMSG_SPACE((sizeof(t_LOOPPROT_MSG_HDR) + sizeof(t_LOOPPROT_REGISTER)));

    new_skb  = alloc_skb(buf_size, GFP_ATOMIC);

    if(!new_skb) {
        printk("br_netlink_loopprot.c:%d %s() errr no mem\n", __LINE__, __FUNCTION__);
        return;
    }

    // init nlh header
    new_nlh              = (struct nlmsghdr *)new_skb->data;
    new_ptr              = NLMSG_DATA(new_nlh);

    new_nlh->nlmsg_len   = NLMSG_SPACE(buf_size);
    new_nlh->nlmsg_pid   = 0;
    new_nlh->nlmsg_flags = 0;
    skb_put(new_skb, buf_size);
    ((t_LOOPPROT_MSG_HDR *)new_ptr)->type = LOOPPROT_MSG_REGISTER;
    ((t_LOOPPROT_MSG_HDR *)new_ptr)->len  = sizeof(t_LOOPPROT_REGISTER);

    // skip past header
    new_ptr += sizeof(t_LOOPPROT_MSG_HDR);
    ((t_LOOPPROT_REGISTER *)new_ptr)->code = LOOPPROT_SUCCESS;

    NETLINK_CB(new_skb).dst_group = 0;
    NETLINK_CB(new_skb).pid       = loopprot_pid;
    ret = netlink_unicast(loopprot_nl_sk, new_skb, loopprot_pid, MSG_DONTWAIT);

    return;
}

static void loopprot_config(struct sk_buff *skb)
{
    struct nlmsghdr *nlh = (struct nlmsghdr *)skb->data;
    unsigned char *ptr;
    int val;

    ptr = NLMSG_DATA(nlh);
    ptr += sizeof(t_LOOPPROT_MSG_HDR);

    val = (int)*(int *)ptr;
    br_loopprot_update(val);

    return;
}


static inline void loopprot_nl_rcv_skb(struct sk_buff *skb)
{
    struct nlmsghdr *nlh = (struct nlmsghdr *)skb->data;
    char *ptr  = NULL;
    unsigned short msg_type;

    if (skb->len >= NLMSG_SPACE(0)) 
    {
        if (nlh->nlmsg_len < sizeof(*nlh) || skb->len < nlh->nlmsg_len)
            return;

        ptr = NLMSG_DATA(nlh);

        msg_type = *(unsigned short *)ptr;
        //printk("\n%s:%d: %d %d\n", __FUNCTION__,__LINE__, skb->len, msg_type);
        switch(msg_type)
        {
            case LOOPPROT_MSG_REGISTER:
                loopprot_nl_process_registration(skb);
                break;

            case LOOPPROT_MSG_LDF_SEND:
                break;
                
            case LOOPPROT_MSG_CONFIG:
                loopprot_config(skb);
                break;

            default:
                printk("loopprot Unknown usr->krnl msg type \n");
        }
    }

    return;
} /* loopprot_nl_rcv_skb */



static int __init loopprot_module_init(void)
{
    printk(KERN_INFO "Initializing loopprot Module in kernel\n");

    loopprot_nl_sk = netlink_kernel_create(&init_net, NETLINK_LOOPPROT, 0, 
                                loopprot_nl_rcv_skb, NULL, THIS_MODULE);

    if(loopprot_nl_sk == NULL) 
    {
        printk("LOOPPROT: failure to create kernel netlink socket\n");
        return -ENOMEM;
    }

    return 0;
}

static void __exit loopprot_module_exit(void)
{
    sock_release(loopprot_nl_sk->sk_socket); 
    printk(KERN_INFO "Stopped loopprot\n");
}

module_init(loopprot_module_init);
module_exit(loopprot_module_exit);
