/*
 * Copyright (C) 2000 Lennert Buytenhek
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "libbridge.h"
#include "libbridge_private.h"

//#if defined (ATI_PRODUCT_ID)
#include <sys/fcntl.h>
# define __set_errno(ev) ((errno) = (ev))
#define unlikely(x)		__builtin_expect(!!(x), 0)
typedef __compar_fn_t comparison_fn_t;

static FILE *fpopen(const char *dir, const char *name)
{
	char path[SYSFS_PATH_MAX];

	snprintf(path, SYSFS_PATH_MAX, "%s/%s", dir, name);
	return fopen(path, "r");
}

int fixedscandir(const char *dir, struct dirent ***namelist,
	int (*selector) (const struct dirent *),
	int (*compar) (const void *, const void *));
int fixedscandir(const char *dir, struct dirent ***namelist,
	int (*selector) (const struct dirent *),
	int (*compar) (const void *, const void *))
{
    DIR *dp = opendir (dir);
    struct dirent *current;
    struct dirent **names = NULL;
    size_t names_size = 0, pos;
    int save;

    if (dp == NULL)
	return -1;

    save = errno;
    __set_errno (0);

    pos = 0;
    while ((current = readdir (dp)) != NULL) {
	int use_it = selector == NULL;

	if (! use_it)
	{
	    use_it = (*selector) (current);
	    /* The selector function might have changed errno.
	     * It was zero before and it need to be again to make
	     * the latter tests work.  */
	    if (! use_it)
		__set_errno (0);
	}
	if (use_it)
	{
	    struct dirent *vnew;
	    size_t dsize;

	    /* Ignore errors from selector or readdir */
	    __set_errno (0);

	    if (unlikely(pos == names_size))
	    {
		struct dirent **new;
		if (names_size == 0)
		    names_size = 10;
		else
		    names_size *= 2;
		new = (struct dirent **) realloc (names,
					names_size * sizeof (struct dirent *));
		if (new == NULL)
		    break;
		names = new;
	    }

	    dsize = &current->d_name[_D_ALLOC_NAMLEN(current)] - (char*)current;
	    vnew = (struct dirent *) malloc (dsize);
	    if (vnew == NULL)
		break;

	    names[pos++] = (struct dirent *) memcpy (vnew, current, dsize);
	}
    }

    if (unlikely(errno != 0))
    {
	save = errno;
	closedir (dp);
	while (pos > 0)
	    free (names[--pos]);
	free (names);
	__set_errno (save);
	return -1;
    }

    closedir (dp);
    __set_errno (save);

    /* Sort the list if we have a comparison function to sort with.  */
    if (compar != NULL)
	qsort (names, pos, sizeof (struct dirent *), (comparison_fn_t) compar);
    *namelist = names;
    return pos;
}
//#endif //defined (ATI_PRODUCT_ID)


int br_socket_fd = -1;

int br_init(void)
{
	if ((br_socket_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		return errno;
	return 0;
}

void br_shutdown(void)
{
	close(br_socket_fd);
	br_socket_fd = -1;
}

/* If /sys/class/net/XXX/bridge exists then it must be a bridge */
static int isbridge(const struct dirent *entry)
{
	char path[SYSFS_PATH_MAX];
	struct stat st;

	snprintf(path, SYSFS_PATH_MAX, SYSFS_CLASS_NET "%s/bridge", entry->d_name);
	return stat(path, &st) == 0 && S_ISDIR(st.st_mode);
}

/*
 * New interface uses sysfs to find bridges
 */
static int new_foreach_bridge(int (*iterator)(const char *name, void *),
			      void *arg)
{
	struct dirent **namelist;
	int i, count = 0;

#if 1 //defined (ATI_PRODUCT_ID)
	count = fixedscandir(SYSFS_CLASS_NET, &namelist, isbridge, alphasort);
#else
	count = scandir(SYSFS_CLASS_NET, &namelist, isbridge, alphasort);
#endif //defined (ATI_PRODUCT_ID)
	if (count < 0)
		return -1;
	for (i = 0; i < count; i++) {
		if (iterator(namelist[i]->d_name, arg))
			break;
	}

	for (i = 0; i < count; i++)
		free(namelist[i]);
	free(namelist);

	return count;
}

/*
 * Old interface uses ioctl
 */
static int old_foreach_bridge(int (*iterator)(const char *, void *), 
			      void *iarg)
{
	int i, ret=0, num;
	char ifname[IFNAMSIZ];
	int ifindices[MAX_BRIDGES];
	unsigned long args[3] = { BRCTL_GET_BRIDGES, 
				 (unsigned long)ifindices, MAX_BRIDGES };

	num = ioctl(br_socket_fd, SIOCGIFBR, args);
	if (num < 0) {
		dprintf("Get bridge indices failed: %s\n",
			strerror(errno));
		return -errno;
	}

	for (i = 0; i < num; i++) {
		if (!if_indextoname(ifindices[i], ifname)) {
			dprintf("get find name for ifindex %d\n",
				ifindices[i]);
			return -errno;
		}

		++ret;
		if(iterator(ifname, iarg)) 
			break;
		
	}

	return ret;

}

/*
 * Go over all bridges and call iterator function.
 * if iterator returns non-zero then stop.
 */
int br_foreach_bridge(int (*iterator)(const char *, void *), 
		     void *arg)
{
	int ret;

	ret = new_foreach_bridge(iterator, arg);
	if (ret <= 0)
		ret = old_foreach_bridge(iterator, arg);

	return ret;
}

/* 
 * Only used if sysfs is not available.
 */
static int old_foreach_port(const char *brname,
			    int (*iterator)(const char *br, const char *port, 
					    void *arg),
			    void *arg)
{
	int i, err, count;
	struct ifreq ifr;
	char ifname[IFNAMSIZ];
	int ifindices[MAX_PORTS];
	unsigned long args[4] = { BRCTL_GET_PORT_LIST,
				  (unsigned long)ifindices, MAX_PORTS, 0 };

	memset(ifindices, 0, sizeof(ifindices));
	strncpy(ifr.ifr_name, brname, IFNAMSIZ);
	ifr.ifr_data = (char *) &args;

	err = ioctl(br_socket_fd, SIOCDEVPRIVATE, &ifr);
	if (err < 0) {
		dprintf("list ports for bridge:'%s' failed: %s\n",
			brname, strerror(errno));
		return -errno;
	}

	count = 0;
	for (i = 0; i < MAX_PORTS; i++) {
		if (!ifindices[i])
			continue;

		if (!if_indextoname(ifindices[i], ifname)) {
			dprintf("can't find name for ifindex:%d\n",
				ifindices[i]);
			continue;
		}

		++count;
		if (iterator(brname, ifname, arg))
			break;
	}

	return count;
}
	
/*
 * Iterate over all ports in bridge (using sysfs).
 */
int br_foreach_port(const char *brname,
		    int (*iterator)(const char *br, const char *port, void *arg),
		    void *arg)
{
	int i, count;
	struct dirent **namelist;
	char path[SYSFS_PATH_MAX];

//#if defined (ATI_PRODUCT_ID)
	snprintf(path, SYSFS_PATH_MAX, SYSFS_CLASS_NET "%s/brif", brname);
	count = fixedscandir(path, &namelist, 0, alphasort);
//#else
//	snprintf(path, SYSFS_PATH_MAX, SYSFS_CLASS_NET "%s/brport", brname);
//	count = fixedscandir(path, &namelist, 0, alphasort);
//#endif //defined (ATI_PRODUCT_ID)
	if (count < 0)
		return old_foreach_port(brname, iterator, arg);

	for (i = 0; i < count; i++) {
//#if defined (ATI_PRODUCT_ID)
		if (namelist[i]->d_name[0]!='.') {
//#endif //defined (ATI_PRODUCT_ID)
			if (iterator(brname, namelist[i]->d_name, arg))
				break;
//#if defined (ATI_PRODUCT_ID)
		}
//#endif //defined (ATI_PRODUCT_ID)
	}
	for (i = 0; i < count; i++)
		free(namelist[i]);
	free(namelist);

	return count;
}


//#if defined (ATI_PRODUCT_ID)
/* Fetch an integer attribute out of sysfs. */
static int fetch_int(const char *dev, const char *name)
{
	FILE *f = fpopen(dev, name);
	int value = -1;

	if (!f) 
		fprintf(stderr, "%s: %s\n", dev, strerror(errno));
	else {
		fscanf(f, "%x", &value);

		fclose(f);
	}
	return value;
}

/*
 * Iterate over all ports in bridge (using sysfs).
 */
int br_portMap(const char *brname,
		    tPortMap **portMap)
{
	int i, count, badcount, port_no, max_port_no=0;
	struct dirent **namelist;
	char path[SYSFS_PATH_MAX];

	snprintf(path, SYSFS_PATH_MAX, SYSFS_CLASS_NET "%s/brif", brname);
//#if defined (ATI_PRODUCT_ID)
	count = fixedscandir(path, &namelist, 0, alphasort);
//#else
//	count = fixedscandir(path, &namelist, 0, alphasort);
//#endif //defined (ATI_PRODUCT_ID)

	for (i = 0; i < count; i++) {
      if (namelist[i]->d_name[0]!='.') {
          snprintf(path, SYSFS_PATH_MAX, SYSFS_CLASS_NET "%s/brport", namelist[i]->d_name);
          port_no = fetch_int(path, "port_no");
          if (port_no >= 0 && port_no > max_port_no) 
          {
            max_port_no = port_no;
          }
      }
	}

	*portMap = malloc(sizeof(tPortMap)*(max_port_no+1));
	if (*portMap) {
		for (i = 0; i < count; i++) {
			if (namelist[i]->d_name[0]!='.') {
				snprintf(path, SYSFS_PATH_MAX, SYSFS_CLASS_NET "%s/brport", namelist[i]->d_name);
				port_no = fetch_int(path, "port_no");
				if (port_no >= 0) {
					snprintf((*portMap)[port_no].portName, sizeof((*portMap)[i].portName), "%s", namelist[i]->d_name); 
//                    strncpy((*portMap)[port_no].portName, namelist[i]->d_name, sizeof((*portMap)[i].portName));
				}
			}
		}
	}

	for (i = 0; i < count; i++)
		free(namelist[i]);
	free(namelist);

	return count;
}
//#endif // defined (ATI_PRODUCT_ID)

