/***********************************************************************
 *
 *  Copyright (c) 2007  Broadcom Corporation
 *  All Rights Reserved
 *
<:label-BRCM:2012:DUAL/GPL:standard

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License, version 2, as published by
the Free Software Foundation (the "GPL").

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.


A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

:>
 *
 ************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <net/if.h>
#ifdef ATI_PRODUCT_CONFIG
#include <strings.h>
#include <sys/stat.h>
#include <assert.h>
#include <errno.h>
#endif

#include "cms.h"
#include "cms_util.h"
#include "cms_msg.h"
#include "cms_boardcmds.h"
#include "cms_boardioctl.h"

#include "bcmTag.h" /* in shared/opensource/include/bcm963xx, for FILE_TAG */
#include "board.h" /* in bcmdrivers/opensource/include/bcm963xx, for BCM_IMAGE_CFE */

#ifdef ATI_PRODUCT_CONFIG
#define MIN_SW_VERSION_NBRS                 2
#define MAX_SW_VERSION_NBRS                 3
#define SW_VERS_ITEM_NUM   2
#define DEFAULT_SW_VERS_STR "AtiBcm-4.1"

#define SMOKING_GUN_TAG   "X_ALLIEDTELESIS_COM_SmokingGun"
#define NVRAM_PARAMS_TAG  "NvramParams"
#define TAG_BEGIN    "<"
#define TAG_END      ">"
#define CLOSURE_TAG  "/"
#define API_START  "AtiProductId "
#define API_END    "\r\n"
#endif
static UBOOL8 matchChipId(const char *strTagChipId, const char *signature2);
CmsRet verifyBroadcomFileTag(FILE_TAG *pTag, int imageLen);
#ifdef ATI_PRODUCT_CONFIG
static CmsRet read_nvram_param(char const * const path, char * const strNvram, const SINT32 strLen);
CmsRet flashImage(char *imagePtr, UINT32 imageLen, CmsImageFormat format, char const * const fileName);
CmsRet sendConfigMsg(const char *imagePtr, UINT32 imageLen, void *msgHandle, CmsMsgType msgType, char const * const fileName);
static CmsRet checkConfigFileCompatibility(const char *imagePtr, UINT32 imageLen);
#else
static CmsRet flashImage(char *imagePtr, UINT32 imageLen, CmsImageFormat format);
static CmsRet sendConfigMsg(const char *imagePtr, UINT32 imageLen, void *msgHandle, CmsMsgType msgType);
#endif
#ifdef SUPPORT_MOD_SW_UPDATE
static void sendStartModupdtdMsg(void *msgHandle);
#endif


/**
 * @return TRUE if the modem's chip id matches that of the image.
 */
UBOOL8 matchChipId(const char *strTagChipId, const char *signature2 __attribute__((unused)))
{
    UINT32 tagChipId = 0;
    UINT32 chipId; 
    UBOOL8 match;

    /* this is the image's chip id */
    tagChipId = (UINT32) strtol(strTagChipId, NULL, 16);
    
    /* get the system's chip id */
    devCtl_getChipId(&chipId);
#if defined(CONFIG_BRCM_6818_ON_6816)
    if (tagChipId == 0x6818)
    {
        match = TRUE;
    }
    else
#endif
    if (tagChipId == chipId)
    {
        match = TRUE;
    }
    else if (tagChipId == BRCM_CHIP_HEX)
    {
        match = TRUE;
    }
    else
    {
        cmsLog_error("Chip Id error.  Image Chip Id = %04x, Board Chip Id = %04x.", tagChipId, chipId);
        match = FALSE;
    }

    return match;
}

// verify the tag of the image
CmsRet verifyBroadcomFileTag(FILE_TAG *pTag, int imageLen)
{
    UINT32 crc;
    int totalImageSize;
    int tagVer, curVer;
    UINT32 tokenCrc, imageCrc;

        
    cmsLog_debug("start of pTag=%p tagversion %02x %02x %02x %02x", pTag,
                  pTag->tagVersion[0], 
                  pTag->tagVersion[1], 
                  pTag->tagVersion[2], 
                  pTag->tagVersion[3]);

    tokenCrc = *((UINT32 *)pTag->tagValidationToken);
    imageCrc = *((UINT32 *)pTag->imageValidationToken);                  
#ifdef DESKTOP_LINUX
    /* assume desktop is running on little endien intel, but the CRC has been
     * written for big endien mips, so swap before compare.
     */
    tokenCrc = htonl(tokenCrc);
    imageCrc = htonl(imageCrc);
#endif
                  
                  
    // check tag validate token first
    crc = CRC_INITIAL_VALUE;
    crc = cmsCrc_getCrc32((UINT8 *) pTag, (UINT32)TAG_LEN-TOKEN_LEN, crc);
    if (crc != tokenCrc)
    {
        /* this function is called even when we are not sure the image is
         * a broadcom image.  So if crc fails, it is not a big deal.  It just
         * means this is not a broadcom image.
         */
        cmsLog_debug("token crc failed, this is not a valid broadcom image");
        cmsLog_debug("calculated crc=0x%x tokenCrc=0x%x", crc, tokenCrc);
        return CMSRET_INVALID_IMAGE;
    }
    cmsLog_debug("header CRC is OK.");

    
    // check imageCrc
    totalImageSize = atoi((char *) pTag->totalImageLen);
    cmsLog_debug("totalImageLen=%d, imageLen=%d, TAG_LEN=%d\n", totalImageSize, imageLen, TAG_LEN);

    if (totalImageSize > (imageLen -TAG_LEN)) {
	 cmsLog_error("invalid size\n");
        return CMSRET_INVALID_IMAGE;
	}
    crc = CRC_INITIAL_VALUE;
    crc = cmsCrc_getCrc32(((UINT8 *)pTag + TAG_LEN), (UINT32) totalImageSize, crc);      
    if (crc != imageCrc)
    {
        /*
         * This should not happen.  We already passed the crc check on the header,
         * so we should pass the crc check on the image.  If this fails, something
         * is wrong.
         */
        cmsLog_error("image crc failed after broadcom header crc succeeded");
        cmsLog_error("calculated crc=0x%x imageCrc=0x%x totalImageSize", crc, imageCrc, totalImageSize); 
        return CMSRET_INVALID_IMAGE;
    }
    cmsLog_debug("image crc is OK");


    tagVer = atoi((char *) pTag->tagVersion);
    curVer = atoi(BCM_TAG_VER);

    if (tagVer != curVer)

    {
       cmsLog_error("Firmware tag version [%d] is not compatible with the current Tag version [%d]", tagVer, curVer);
       return CMSRET_INVALID_IMAGE;
    }
    cmsLog_debug("tarVer=%d, curVar=%d", tagVer, curVer);

    if (!matchChipId((char *) pTag->chipId, (char *) pTag->signiture_2))
    {
       cmsLog_error("chipid check failed");
       return CMSRET_INVALID_IMAGE;
    }

#ifdef ATI_PRODUCT_CONFIG
    {
       char tmpStr[BOARD_ID_LEN] = {'\0'};

       if (strlen(pTag->boardId) > 0 && CMSRET_SUCCESS == read_nvram_param(ATI_BOARD_ID_FILE, tmpStr, sizeof(tmpStr)-1))
       {
          if(!strncmp("96", pTag->boardId, 2))
          {
             cmsLog_notice("pTag->boardId begins with '96' - skipping family check");
             /*
              * This is the 'do nothing' case. It is here to preserve compatibility
              * with release 4.0 as in that release the image header boardId field
              * contained a Bcm board ID value ('96368VVW' or '96816R' - for iMG1500
              * & iMG2500 respectively) rather than our board ID/family data.
              *
              * In other words; we skip the product family check if the image
              * that we've downloaded contains a Bcm board ID value.
              */
          }
          else
          {
             if(strncasecmp(tmpStr, pTag->boardId, BOARD_ID_LEN))
             {
                cmsLog_error("iMG family check failed (%s versus %s)", pTag->boardId, tmpStr);
                return CMSRET_INVALID_IMAGE;
             }
          }
       }
    }

    {
        char currentSwVersion[BUFLEN_32] = {0};
        char minimumSwVersion[BUFLEN_32] = {0};
        const char *swVersion[SW_VERS_ITEM_NUM] = {currentSwVersion, minimumSwVersion};
        int sscanfRet[SW_VERS_ITEM_NUM] = {0, 0};
        int i = 0;
        UINT32 majVersion[SW_VERS_ITEM_NUM] = {0, 0};
        UINT32 minVersion[SW_VERS_ITEM_NUM] = {0, 0};
        UINT32 bldVersion[SW_VERS_ITEM_NUM] = {0, 0};
        UINT32 cmpVersion[SW_VERS_ITEM_NUM] = {0, 0};

        /* get sw versions to be compared */
        (void)strncpy(currentSwVersion, pTag->imageVersion, sizeof(currentSwVersion)-1);
        (void)read_nvram_param(ATI_SW_VERSION_COMP_FILE, minimumSwVersion, sizeof(minimumSwVersion)-1);

        /* 
         *  
         * Software version has one of the following syntax:
         *  
         * AtiBcm-a.b, AtiBcm-a.b.c
         *  
         * where: 
         *  
         * a: main version 
         * b: minor version
         * c: build version
         *  
         * In order to understand whether the software downloaded has a valid sw version, 
         * you have to start comparing from the main version to the build id. 
         * The sw version of the downloaded file must be greater or equal the minimum allowed one.
         *  
         */

        /* collect the sw version number depending on the string format */
        cmsLog_debug("Downloaded sw version: %s, minimum sw version: %s", swVersion[0], swVersion[1]);

        for (i = 0; i < SW_VERS_ITEM_NUM; i++)
        {
            /* AtiBcm-a.b.c */
            if (MAX_SW_VERSION_NBRS != (sscanfRet[i] = sscanf(swVersion[i], "AtiBcm-%d.%d.%d", &majVersion[i], &minVersion[i], &bldVersion[i])))
            {
                /* AtiBcm-a.b */
                sscanfRet[i] = sscanf(swVersion[i], "AtiBcm-%d.%d", &majVersion[i], &minVersion[i]);
            }

            /* wrong format detected : assume DEFAULT_SW_VERS_STR as software version */
            if (sscanfRet[i] < MIN_SW_VERSION_NBRS)
            {
               cmsLog_notice("Wrong format of %s sw version (%s). Assumed to be: %s.", ((0==i) ? "downloaded" : "minimum"), swVersion[i], DEFAULT_SW_VERS_STR);
               strncpy((char*)swVersion[i], DEFAULT_SW_VERS_STR, sizeof(minimumSwVersion)-1);
               sscanfRet[i] = sscanf(swVersion[i], "AtiBcm-%d.%d", &majVersion[i], &minVersion[i]);
            }
        }

        /* check sw version vs minimum one */
        if ( (sscanfRet[0] >= MIN_SW_VERSION_NBRS) && (sscanfRet[1] >= MIN_SW_VERSION_NBRS) )
        {
           cmpVersion[0] = majVersion[0]*1000000 + minVersion[0]*1000 + bldVersion[0];
           cmpVersion[1] = majVersion[1]*1000000 + minVersion[1]*1000 + bldVersion[1];

           if (cmpVersion[0] < cmpVersion[1])
           {
               cmsLog_error("Downloaded sw version %s is older than earliest supported version %s", swVersion[0], swVersion[1]);

               return CMSRET_DU_UPDATE_DOWNGRADE_NOT_ALLOWED;
           }
        }
        else
        {
            /* either the downloaded software version or the earliest supported one has a wrong format */
            if (sscanfRet[0] < MIN_SW_VERSION_NBRS)
            {
                cmsLog_error("Wrong format of downloaded sw version: %s", swVersion[0]);
            }
            if (sscanfRet[1] < MIN_SW_VERSION_NBRS)
            {
                cmsLog_error("Wrong format of earliest supported version: %s", swVersion[1]);
            }
        }
    }
#endif

    cmsLog_debug("Good broadcom image");

    return CMSRET_SUCCESS;
}


// depending on the image type, do the brcm image or whole flash image
#ifdef ATI_PRODUCT_CONFIG
CmsRet flashImage(char *imagePtr, UINT32 imageLen, CmsImageFormat format, char const * const fileName)
#else
CmsRet flashImage(char *imagePtr, UINT32 imageLen, CmsImageFormat format)
#endif
{
    FILE_TAG *pTag = (FILE_TAG *) imagePtr;
    int cfeSize, rootfsSize, kernelSize, noReboot;
    unsigned long cfeAddr, rootfsAddr, kernelAddr;
    CmsRet ret;
    
#ifdef ATI_PRODUCT_CONFIG    
    if(fileName != NULL)
    {
        strncpy(pTag->imageName, fileName, IMAGE_NAME_LEN-1);
    }
#endif
    
    // a separate variable, 'noReboot'.
    if( (format & CMS_IMAGE_FORMAT_PART1) == CMS_IMAGE_FORMAT_PART1 )
    {
        noReboot = ((format & CMS_IMAGE_FORMAT_NO_REBOOT) == 0)
            ? FLASH_PART1_REBOOT : FLASH_PART1_NO_REBOOT;
    }
    else if( (format & CMS_IMAGE_FORMAT_PART2) == CMS_IMAGE_FORMAT_PART2 )
    {
        noReboot = ((format & CMS_IMAGE_FORMAT_NO_REBOOT) == 0)
            ? FLASH_PART2_REBOOT : FLASH_PART2_NO_REBOOT;
    }
    else
        noReboot = ((format & CMS_IMAGE_FORMAT_NO_REBOOT) == 0)
            ? FLASH_PARTDEFAULT_REBOOT : FLASH_PARTDEFAULT_NO_REBOOT;

    // Clear unneeded partition and reboot bits from 'format' variable
    format &= ~(CMS_IMAGE_FORMAT_NO_REBOOT | CMS_IMAGE_FORMAT_PART1 |
       CMS_IMAGE_FORMAT_PART2); 

    if (format != CMS_IMAGE_FORMAT_FLASH && format != CMS_IMAGE_FORMAT_BROADCOM)
    {
       cmsLog_error("invalid image format %d", format);
       return CMSRET_INVALID_IMAGE;
    }

    if (format == CMS_IMAGE_FORMAT_FLASH)  
    {
        cmsLog_notice("Flash whole image...");
        // Pass zero for the base address of whole image flash. It will be filled by kernel code
        // was sysFlashImageSet
        ret = devCtl_boardIoctl(BOARD_IOCTL_FLASH_WRITE,
                                BCM_IMAGE_WHOLE,
                                imagePtr,
                                imageLen-TOKEN_LEN,
                                0, 0);
        if (ret != CMSRET_SUCCESS)
        {
           cmsLog_error("Failed to flash whole image");
           return CMSRET_IMAGE_FLASH_FAILED;
        }
        else
        {
           return CMSRET_SUCCESS;
        }
    }

    /* this must be a broadcom format image */
    // check imageCrc
    cfeSize = rootfsSize = kernelSize = 0;

    // check cfe's existence
    cfeAddr = (unsigned long) strtoul((char *) pTag->cfeAddress, NULL, 10);
    cfeSize = atoi((char *) pTag->cfeLen);
    // check kernel existence
    kernelAddr = (unsigned long) strtoul((char *) pTag->kernelAddress, NULL, 10);
    kernelSize = atoi((char *) pTag->kernelLen);
    // check root filesystem existence
    rootfsAddr = (unsigned long) strtoul((char *) pTag->rootfsAddress, NULL, 10);
    rootfsSize = atoi((char *) pTag->rootfsLen);
    cmsLog_debug("cfeSize=%d kernelSize=%d rootfsSize=%d", cfeSize, kernelSize, rootfsSize);
    
    if (cfeAddr) 
    {
        printf("Flashing CFE...\n");

        ret = devCtl_boardIoctl(BOARD_IOCTL_FLASH_WRITE,
                                BCM_IMAGE_CFE,
                                imagePtr+TAG_LEN,
                                cfeSize,
                                (int) cfeAddr, 0);
        if (ret != CMSRET_SUCCESS)
        {
            cmsLog_error("Failed to flash CFE");
            return CMSRET_IMAGE_FLASH_FAILED;
        }
    }

    if (rootfsAddr && kernelAddr) 
    {
        char *tagFs = imagePtr;

        // tag is alway at the sector start of fs
        if (cfeAddr)
        {
            tagFs = imagePtr + cfeSize;       // will trash cfe memory, but cfe is already flashed
            memcpy(tagFs, imagePtr, TAG_LEN);
        }

#ifdef ATI_PRODUCT_CONFIG
        cmsLog_debug("Flashing root file system and kernel...");

        /*
         * Give buffered logging a chance to complete before
         * lower-level Bcm flash write helper functions splatter
         * the console with unbuffered printfs.
         */
        sleep(1);
#else
        printf("Flashing root file system and kernel...\n");
#endif
        /* only the buf pointer and length is needed, the offset parameter
         * was present in the legacy code, but is not used. */
        ret = devCtl_boardIoctl(BOARD_IOCTL_FLASH_WRITE,
                                BCM_IMAGE_FS,
                                tagFs,
                                TAG_LEN+rootfsSize+kernelSize,
                                noReboot, 0);
        if (ret != CMSRET_SUCCESS)
        {
            cmsLog_error("Failed to flash root file system and kernel");
            return CMSRET_IMAGE_FLASH_FAILED;
        }
    }

    cmsLog_notice("Image flash done.");
    
    return CMSRET_SUCCESS;
}


UINT32 cmsImg_getImageFlashSize(void)
{
   UINT32 flashSize=0;
   CmsRet ret;
   
   ret = devCtl_boardIoctl(BOARD_IOCTL_FLASH_READ,
                           FLASH_SIZE,
                           0, 0, 0, &flashSize);
   if (ret != CMSRET_SUCCESS)
   {
      cmsLog_error("Could not get flash size, return 0");
      flashSize = 0;
   }
   
   return flashSize;
}


UINT32 cmsImg_getBroadcomImageTagSize(void)
{
   return TOKEN_LEN;
}


UINT32 cmsImg_getConfigFlashSize(void)
{
   UINT32 realSize;

   realSize = cmsImg_getRealConfigFlashSize();

#ifdef COMPRESSED_CONFIG_FILE   
   /*
    * A multiplier of 2 is now too small for some of the big voice and WLAN configs,   
    * so allow for the possibility of 4x compression ratio.  In a recent test on the
    * 6368 with wireless enabled, I got a compression ratio of 3.5.
    * The real test comes in management.c: after we do the
    * compression, writeValidatedConfigBuf will verify that the compressed buffer can
    * fit into the flash.
    * A 4x multiplier should be OK for small memory systems such as the 6338.
    * The kernel does not allocate physical pages until they are touched.
    * However, allocating an overly large buffer could be rejected immediately by the
    * kernel because it does not know we don't actually plan to use the entire buffer.
    * So if this is a problem on the 6338, we could improve this algorithm to
    * use a smaller factor on low memory systems.
    */
#ifdef ATI_PRODUCT_CONFIG
   /*
    * With large configs (c. 405kb) we're seeing compression ratios of 5.6:1...
    */
   realSize = realSize * 8;
#else
   realSize = realSize * 4;
#endif // ATI_PRODUCT_CONFIG
#endif

   return realSize;
}


UINT32 cmsImg_getRealConfigFlashSize(void)
{
   CmsRet ret;
   UINT32 size=0;

   ret = devCtl_boardIoctl(BOARD_IOCTL_GET_PSI_SIZE, 0, NULL, 0, 0, (void *)&size);
   if (ret != CMSRET_SUCCESS)
   {
      cmsLog_error("boardIoctl to get config flash size failed.");
      size = 0;
   }

   return size;
}


UBOOL8 cmsImg_willFitInFlash(UINT32 imageSize)
{
   UINT32 flashSize;
   
   flashSize = cmsImg_getImageFlashSize();

   cmsLog_debug("flash size is %u bytes, imageSize=%u bytes", flashSize, imageSize);
                     
   return (flashSize > (imageSize + CMS_IMAGE_OVERHEAD));
}


UBOOL8 cmsImg_isBackupConfigFlashAvailable(void)
{
   static UBOOL8 firstTime=TRUE;
   static UBOOL8 avail=FALSE;
   UINT32 size=0;
   CmsRet ret;

   if (firstTime)
   {
      ret = devCtl_boardIoctl(BOARD_IOCTL_GET_BACKUP_PSI_SIZE, 0, NULL, 0, 0, (void *)&size);
      if (ret == CMSRET_SUCCESS && size > 0)
      {
         avail = TRUE;
      }

      firstTime = FALSE;
   }

   return avail;
}


UBOOL8 cmsImg_isConfigFileLikely(const char *buf)
{
   const char *header = "<?xml version";
   const char *dslCpeConfig = "<DslCpeConfig";
   UINT32 len, i;
   UBOOL8 likely=FALSE;
   
   if (strncmp(buf, "<?xml version", strlen(header)) == 0)
   {
      len = strlen(dslCpeConfig);
      for (i=20; i<50 && !likely; i++)
      {
         if (strncmp(&(buf[i]), dslCpeConfig, len) == 0)
         {
            likely = TRUE;
         }
      }
   }
   
   cmsLog_debug("returning likely=%d", likely);
   
   return likely;
}


/** General entry point for writing the image.
 *  The image can be a flash image or a config file.
 *  This function will first determine the image type, which has the side
 *  effect of validating it.
 */
#ifdef ATI_PRODUCT_CONFIG
CmsRet cmsImg_writeImage(char *imagePtr, UINT32 imageLen, void *msgHandle, char const * const fileName)
#else
CmsRet cmsImg_writeImage(char *imagePtr, UINT32 imageLen, void *msgHandle)
#endif
{
   CmsImageFormat format;
   CmsRet ret;
   
   if ((format = cmsImg_validateImage(imagePtr, imageLen, msgHandle)) == CMS_IMAGE_FORMAT_INVALID)
   {
      ret = CMSRET_INVALID_IMAGE;
   }
   else
   {
#ifdef ATI_PRODUCT_CONFIG
      ret = cmsImg_writeValidatedImage(imagePtr, imageLen, format, msgHandle, fileName);
#else
      ret = cmsImg_writeValidatedImage(imagePtr, imageLen, format, msgHandle);
#endif
   }

   return ret;
}


#ifdef ATI_PRODUCT_CONFIG
CmsRet cmsImg_writeValidatedImage(char *imagePtr, UINT32 imageLen, CmsImageFormat format, void *msgHandle, char const * const fileName)
#else
CmsRet cmsImg_writeValidatedImage(char *imagePtr, UINT32 imageLen, CmsImageFormat format, void *msgHandle)
#endif
{
   CmsRet ret=CMSRET_SUCCESS;
   
   switch(format & ~(CMS_IMAGE_FORMAT_NO_REBOOT | CMS_IMAGE_FORMAT_PART1 | CMS_IMAGE_FORMAT_PART2))
   {
   case CMS_IMAGE_FORMAT_BROADCOM:
   case CMS_IMAGE_FORMAT_FLASH:
      // BcmNtwk_unInit(); mwang_todo: is it important to let Wireless do some
      // uninit before we write the flash image?
#ifdef ATI_PRODUCT_CONFIG
      ret = flashImage(imagePtr, imageLen, format, fileName);
#else
      ret = flashImage(imagePtr, imageLen, format);
#endif
      break;
      
   case CMS_IMAGE_FORMAT_XML_CFG:
#ifdef ATI_PRODUCT_CONFIG
      ret = sendConfigMsg(imagePtr, imageLen, msgHandle, CMS_MSG_WRITE_CONFIG_FILE, fileName);
#else
      ret = sendConfigMsg(imagePtr, imageLen, msgHandle, CMS_MSG_WRITE_CONFIG_FILE);
#endif
      if (ret == CMSRET_SUCCESS)
      {
         /*
          * Emulate the behavior of the driver when a flash image is written.
          * When we write a config image, also request immediate reboot
          * because we don't want to let any other app save the config file
          * to flash, thus wiping out what we just written.
          */
         cmsLog_debug("config file download written, request reboot");
#if !defined(ATI_PRODUCT_CONFIG)
         cmsUtil_sendRequestRebootMsg(msgHandle);
#endif
      }
      break;
      
#ifdef SUPPORT_MOD_SW_UPDATE
   case CMS_IMAGE_FORMAT_MODSW_PKG:
   {
      char filename[BUFLEN_1024]={0};

      cmsFil_removeDir(SW_UPDATE_DATA_DIR);
      cmsFil_makeDir(SW_UPDATE_DATA_DIR);

      snprintf(filename, sizeof(filename)-1, "%s/%s",
                         SW_UPDATE_DATA_DIR, SW_UPDATE_PKG_FILE);
      cmsFil_writeBufferToFile(filename, (UINT8 *)imagePtr, imageLen);
      /*
       * modupdtd will unpack the sw package, and then send a graceful
       * shutdown msg to smd.
       */
      sendStartModupdtdMsg(msgHandle);
      break;
   }
#endif

   default:
       cmsLog_error("Unrecognized image format=%d", format);
       ret = CMSRET_INVALID_IMAGE;
       break;
    }
   
   return ret;
}

 
CmsImageFormat cmsImg_validateImage(const char *imageBuf, UINT32 imageLen, void *msgHandle)
{
   CmsImageFormat result = CMS_IMAGE_FORMAT_INVALID;
#ifdef ATI_PRODUCT_CONFIG
   CmsRet ret = CMSRET_SUCCESS;
#else
   CmsRet ret;
#endif
   
   if (imageBuf == NULL)
   {
      return CMS_IMAGE_FORMAT_INVALID;
   }
   
   if (imageLen > CMS_CONFIG_FILE_DETECTION_LENGTH &&
       cmsImg_isConfigFileLikely(imageBuf))
   {
      cmsLog_debug("possible CMS XML config file format detected");
#ifdef ATI_PRODUCT_CONFIG
      /* Check if the config file has to be accepted of rejected */
      if ((ret = checkConfigFileCompatibility(imageBuf, imageLen)) != CMSRET_SUCCESS)
      { 
         cmsLog_error("CMS XML config file not compatible");
         return CMS_IMAGE_FORMAT_INVALID;
      }
#else
      ret = sendConfigMsg(imageBuf, imageLen, msgHandle, CMS_MSG_VALIDATE_CONFIG_FILE);
#endif
      if (ret == CMSRET_SUCCESS)
      { 
         cmsLog_debug("CMS XML config format verified.");
         return CMS_IMAGE_FORMAT_XML_CFG;
      }
   } 
   
   cmsLog_debug("not a config file");
   
#ifdef SUPPORT_MOD_SW_UPDATE
   if (cmsImg_isModSwPkg((unsigned char *) imageBuf, imageLen))
   {
      cmsLog_debug("detected modular sw pkg format!");
      return CMS_IMAGE_FORMAT_MODSW_PKG;
   }

   cmsLog_debug("not a modular sw pkg");
#endif

   if ((imageLen > sizeof(FILE_TAG)) 
#ifdef ATI_PRODUCT_CONFIG
        && ((ret = verifyBroadcomFileTag((FILE_TAG *) imageBuf, imageLen)) == CMSRET_SUCCESS))
#else
        && (verifyBroadcomFileTag((FILE_TAG *) imageBuf, imageLen) == CMSRET_SUCCESS))
#endif
   {
      UINT32 maxLen;
      
      /* Found a valid Broadcom defined TAG record at the beginning of the image */
      cmsLog_debug("Broadcom format verified.");
      maxLen = cmsImg_getImageFlashSize() + cmsImg_getBroadcomImageTagSize();
      if (imageLen > maxLen)
      {
         cmsLog_error("broadcom image is too large for flash, got %u, max %u", imageLen, maxLen);
      }
      else
      {
         result = CMS_IMAGE_FORMAT_BROADCOM;
      }
   }
#ifdef ATI_PRODUCT_CONFIG
   else if (CMSRET_INVALID_IMAGE == ret || CMSRET_DU_UPDATE_DOWNGRADE_NOT_ALLOWED == ret)
   {
      /* CMSRET_DU_UPDATE_DOWNGRADE_NOT_ALLOWED is returned when the image is valid, but too old to be used on
       * this iMG (i.e. it's earlier than the version stated in AtiSwVersionCompatibility
       */
      result = CMS_IMAGE_FORMAT_INVALID;
   }
#endif
   else
   {
      /* if It is not a Broadcom flash format file.  Now check if it is a
       * flash image format file.  A flash image format file must have a
       * CRC at the end of the image.
       */
      UINT32 crc = CRC_INITIAL_VALUE;
      UINT32 imageCrc;
      UINT8 *crcPtr;
      
      if (imageLen > TOKEN_LEN)
      {
         crcPtr = (UINT8 *) (imageBuf + (imageLen - TOKEN_LEN));
         /*
          * CRC may not be word aligned, so extract the bytes out one-by-one.
          * Whole image CRC is calculated, then htonl, then written out using
          * fwrite (see addvtoken.c in hostTools).  Because of the way we are
          * extracting the CRC here, we don't have to swap for endieness when
          * doing compares on desktop Linux and modem (?).
          */
         imageCrc = (crcPtr[0] << 24) | (crcPtr[1] << 16) | (crcPtr[2] << 8) | crcPtr[3];
      
         crc = cmsCrc_getCrc32((unsigned char *) imageBuf, imageLen - TOKEN_LEN, crc);      
         if (crc == imageCrc)
         {
            UINT32 maxLen;
          
            cmsLog_debug("Whole flash image format [%ld bytes] verified.", imageLen);
            maxLen = cmsImg_getImageFlashSize();
            if (imageLen > maxLen)
            {
               cmsLog_error("whole image is too large for flash, got %u, max %u", imageLen, maxLen);
            }
            else
            {
               result = CMS_IMAGE_FORMAT_FLASH;
            }
         }
         else
         {
#if defined(EPON_SDK_BUILD)
            cmsLog_debug("Could not determine image format [%d bytes]", imageLen);
#else
            cmsLog_error("Could not determine image format [%d bytes]", imageLen);
#endif
            cmsLog_debug("calculated crc=0x%x image crc=0x%x", crc, imageCrc);
         }
      }
   }

   cmsLog_debug("returning image format %d", result);

   return result;
}
#ifdef ATI_PRODUCT_CONFIG
CmsRet sendConfigMsg(const char *imagePtr, UINT32 imageLen, void *msgHandle, CmsMsgType msgType, char const * const fileName)
#else
CmsRet sendConfigMsg(const char *imagePtr, UINT32 imageLen, void *msgHandle, CmsMsgType msgType)
#endif
{
   char *buf=NULL;
   char *body=NULL;
   CmsMsgHeader *msg;
   CmsRet ret;
#ifdef ATI_PRODUCT_CONFIG
   char *name=NULL;

   assert(fileName != NULL);
   if ((buf = cmsMem_alloc(sizeof(CmsMsgHeader) + imageLen + strlen(fileName) + 1, ALLOC_ZEROIZE)) == NULL)
   {
      cmsLog_error("failed to allocate %d bytes for msg 0x%x", 
                   sizeof(CmsMsgHeader) + imageLen + strlen(fileName) + 1, msgType);
      return CMSRET_RESOURCE_EXCEEDED;
   }
   
   msg = (CmsMsgHeader *) buf;
   name = (char *) (msg + 1);
   body = name + strlen(fileName) + 1;
   strcpy(name, fileName);
#else
   if ((buf = cmsMem_alloc(sizeof(CmsMsgHeader) + imageLen, ALLOC_ZEROIZE)) == NULL)
   {
      cmsLog_error("failed to allocate %d bytes for msg 0x%x", 
                   sizeof(CmsMsgHeader) + imageLen, msgType);
      return CMSRET_RESOURCE_EXCEEDED;
   }
   
   msg = (CmsMsgHeader *) buf;
   body = (char *) (msg + 1);
#endif
    
   msg->type = msgType;
   msg->src = cmsMsg_getHandleEid(msgHandle);
   msg->dst = EID_SMD;
   msg->flags_request = 1;
   msg->dataLength = imageLen;
#ifdef ATI_PRODUCT_CONFIG
   msg->dataLength = msg->dataLength + strlen(fileName) + 1;
#endif   
   memcpy(body, imagePtr, imageLen);

   ret = cmsMsg_sendAndGetReply(msgHandle, msg);
   
   cmsMem_free(buf);
   
   return ret;
}


void cmsImg_sendLoadStartingMsg(void *msgHandle, const char *connIfName)
{
   CmsMsgHeader *msg;
   char *data;
   void *msgBuf;
   UINT32 msgDataLen=0;

   /* for the msg and the connIfName */
   if (connIfName)
   {
      msgDataLen = strlen(connIfName) + 1;
      msgBuf = cmsMem_alloc(sizeof(CmsMsgHeader) + msgDataLen, ALLOC_ZEROIZE);
   } 
   else
   {
#ifndef ATI_PRODUCT_CONFIG
      cmsLog_error("msg without connIfName");
#endif
      msgBuf = cmsMem_alloc(sizeof(CmsMsgHeader), ALLOC_ZEROIZE);
   }
   
   msg = (CmsMsgHeader *)msgBuf;
   msg->src = cmsMsg_getHandleEid(msgHandle);
   msg->dst = EID_SMD;
   msg->flags_request = 1;
   msg->type = CMS_MSG_LOAD_IMAGE_STARTING;

   if (connIfName)
   {
      data = (char *) (msg + 1);
      msg->dataLength = msgDataLen;
      memcpy(data, (char *)connIfName, msgDataLen);      
   }
   
   cmsLog_debug("connIfName=%s", connIfName);

   cmsMsg_sendAndGetReply(msgHandle, msg);

   CMSMEM_FREE_BUF_AND_NULL_PTR(msgBuf);

}

#if defined(ATI_PRODUCT_CONFIG)
static CmsRet read_nvram_param(char const * const path, char * const strNvram, const SINT32 strLen)
{
   FILE* nvamFs = NULL;
   CmsRet ret = CMSRET_SUCCESS;

   memset(strNvram, 0, strLen);
   if((nvamFs = fopen(path, "r")) != NULL)
   {
      /*
       * The following test protects us when we try to read an empty file
       */
      if(fgets(strNvram, strLen, nvamFs) != NULL)
      {
         if('\n' == strNvram[strlen(strNvram) - 1])
         {
            /*
             * Strip the trailing \n that fgets() has appended to the content
             */
            strNvram[strlen(strNvram) - 1] = '\0';
         }
      }

      cmsLog_debug("Read file: %s and cooked content: '%s'", path, strNvram);
      fclose(nvamFs);
   }
   else
   {
      cmsLog_debug("Failed to open file %s ", path);
      ret =  CMSRET_OPEN_FILE_ERROR;
   }
   return ret;
}


static CmsRet checkConfigFileCompatibility(const char *imagePtr, UINT32 imageLen)
{
    CmsRet ret = CMSRET_INVALID_CONFIG_FILE;
    char tagString[BUFLEN_64] = {0};
    char atiProductId[BUFLEN_64] = {0};
    UBOOL8 smokingGunObjFound = FALSE;
    UBOOL8 nvParamFound = FALSE;
    UBOOL8 atiProductIdFound = FALSE;
    char* tmpImg = NULL;
    char* from = NULL;
    char* to = NULL;

    /* treat the image as a string */
    if (NULL == (tmpImg = cmsMem_alloc(imageLen + 1, ALLOC_ZEROIZE)))
    {
        cmsLog_error("Failed allocating memory in %s", __FUNCTION__);
        return ret;
    }
    else
    {
        memset((void*)tmpImg, 0, imageLen+1);
        strncpy(tmpImg, imagePtr, imageLen);
    }

    /* get the model id from nvram */
    (void)read_nvram_param(ATI_PRODUCT_ID_FILE, atiProductId, sizeof(atiProductId)-1);

    /* get AtiProductId from config file */

    /* search for the smoking gun object start */
    from = tmpImg;
    (void)snprintf(tagString, sizeof(tagString)-1, "%s%s%s", TAG_BEGIN, SMOKING_GUN_TAG, TAG_END);
    if (NULL != (from = strstr(from, tagString)))
    {
        /* search for the smoking gun object end */
        (void)snprintf(tagString, sizeof(tagString)-1, "%s%s%s%s", TAG_BEGIN, CLOSURE_TAG, SMOKING_GUN_TAG, TAG_END);
        if (NULL != (to = strstr(from, tagString)))
        {
            /* search for the NvParam parameter start */
            smokingGunObjFound = TRUE;
            *to = 0;
            memset((void*)tagString, 0, sizeof(tagString));
            (void)snprintf(tagString, sizeof(tagString)-1, "%s%s%s", TAG_BEGIN, NVRAM_PARAMS_TAG, TAG_END);
            if (NULL != (from = strstr(from, tagString)))
            {
                /* search for the NvParam parameter end */
                from += strlen(tagString);
                memset((void*)tagString, 0, sizeof(tagString));
                (void)snprintf(tagString, sizeof(tagString)-1, "%s%s%s%s", TAG_BEGIN, CLOSURE_TAG, NVRAM_PARAMS_TAG, TAG_END);
                if (NULL != (to = strstr(from, tagString)))
                {
                    /* search for the AtiProductId parameter */
                    nvParamFound = TRUE;
                    *to = 0;
                    memset((void*)tagString, 0, sizeof(tagString));
                    (void)strncpy(tagString, API_START, sizeof(tagString)-1);
                    if (NULL != (from = strstr(from, tagString)))
                    {
                        /* search for the AtiProductId string end */
                        from += strlen(tagString);
                        memset((void*)tagString, 0, sizeof(tagString));
                        (void)strncpy(tagString, API_END, sizeof(tagString)-1);

                        /* 
                         * Search for end of line. 
                         *  
                         * Note that it can be:
                         *  - linux file: \n
                         *  - DOS file: \r\n
                         *  - MAC file: \r
                         * 
                         */
                        if (NULL != (to = strpbrk(from, tagString)))
                        {
                            /* got the string */
                            atiProductIdFound = TRUE;
                            *to = 0;
                        }  
                    }
                }
            }
        }
    }

    /* analize the data found */
    if (!smokingGunObjFound)
    {
        /* either it's a default config file or a bootstrap one: accept it */
        cmsLog_debug("Default config file");
        ret = CMSRET_SUCCESS;
    }
    else
    {
        /* 
         * Once the X_ALLIEDTELESIS_COM_SmokingGunObject has been found,
         *  - the NvParams parameter can be missing
         *  - the AtiProductId can be missing
         * In both cases the config file is accepted. 
         * In case both are present, a further comparation has to be done 
         * because it could be a config file coming from a different board. 
         */
        if (!nvParamFound || !atiProductIdFound)
        {
            /* it's a config file with either NvParams or AtiProductId deleted */
            cmsLog_debug("Config file accepted");
            ret = CMSRET_SUCCESS;
        }
        else
        {
            /* compare the model id from nvram with the one in the config file */
            if (0 == strncmp(from, atiProductId, sizeof(atiProductId)))
            {
                /* this file comes from a board of the same model */
                cmsLog_debug("Config file comes from a compatible model");
                ret = CMSRET_SUCCESS;
            }
            else
            {
                /* this file comes from a board of a different model */
                cmsLog_error("Config file comes from a not compatible model (%s)", from);
                ret = CMSRET_INVALID_CONFIG_FILE;
            }
        }
    }

    cmsMem_free((void*)tmpImg);
    return ret;
}
#endif

void cmsImg_sendLoadDoneMsg(void *msgHandle)
{
   CmsMsgHeader msg = EMPTY_MSG_HEADER;

   msg.type = CMS_MSG_LOAD_IMAGE_DONE;
   msg.src = cmsMsg_getHandleEid(msgHandle);
   msg.dst = EID_SMD;
   msg.flags_request = 1;
   
   cmsMsg_sendAndGetReply(msgHandle, &msg);
}


#ifdef SUPPORT_MOD_SW_UPDATE
void sendStartModupdtdMsg(void *msgHandle)
{
   CmsMsgHeader msg = EMPTY_MSG_HEADER;

   msg.type = CMS_MSG_START_APP;
   msg.src = cmsMsg_getHandleEid(msgHandle);
   msg.dst = EID_SMD;
   msg.wordData = EID_MODUPDTD;
   msg.flags_request = 1;

   cmsMsg_sendAndGetReply(msgHandle, &msg);
}
#endif


/* using a cmsUtil_ prefix because this can be used in non-upload scenarios */
void cmsUtil_sendRequestRebootMsg(void *msgHandle)
{
   CmsMsgHeader msg = EMPTY_MSG_HEADER;

   msg.type = CMS_MSG_REBOOT_SYSTEM;
   msg.src = cmsMsg_getHandleEid(msgHandle);
   msg.dst = EID_SMD;
   msg.flags_request = 1;

   cmsMsg_sendAndGetReply(msgHandle, &msg);
}

#ifdef ATI_PRODUCT_CONFIG
void cmsUtil_sendDiscoveryRebootMsg(void *msgHandle)
{
   CmsMsgHeader msg = EMPTY_MSG_HEADER;
   CmsRet ret = CMSRET_SUCCESS;

   msg.type = CMS_MSG_DISCOVERY_REBOOT_SYSTEM;
   msg.src = cmsMsg_getHandleEid(msgHandle);
   msg.dst = EID_SSK;
   msg.flags_request = 0;
   msg.flags_response = 1;

   if ((ret = cmsMsg_send(msgHandle, &msg)) != CMSRET_SUCCESS)
   {
      cmsLog_error("msg send failed, ret = %d",ret);
   }

   return;
}

/* using a cmsUtil_ prefix because this can be used in non-upload scenarios */
void cmsUtil_sendRequestRebootMsgFromSwupdate(void *msgHandle)
{
   CmsMsgHeader msg = EMPTY_MSG_HEADER;

   msg.type = CMS_MSG_REBOOT_SYSTEM_FROM_SWUPDATE;
   msg.src = cmsMsg_getHandleEid(msgHandle);
   msg.dst = EID_SMD;
   msg.flags_request = 1;

   cmsMsg_sendAndGetReply(msgHandle, &msg);
}
#endif


CmsRet cmsImg_saveIfNameFromSocket(SINT32 socketfd, char *connIfName)
{
  
   SINT32 i = 0;
   SINT32 fd = 0;
   SINT32 numifs = 0;
   UINT32 bufsize = 0;
   struct ifreq *all_ifr = NULL;
   struct ifconf ifc;
   struct sockaddr local_addr;
   socklen_t local_len = sizeof(struct sockaddr_in);

   if (socketfd < 0 || connIfName == NULL)
   {
      cmsLog_error("cmsImg_saveIfNameFromSocket: Invalid parameters: socket=%d, connIfName=%s", socketfd, connIfName);
      return CMSRET_INTERNAL_ERROR;
   }
   memset(&ifc, 0, sizeof(struct ifconf));
   memset(&local_addr, 0, sizeof(struct sockaddr));
   
   if (getsockname(socketfd, &local_addr,&local_len) < 0) 
   {
      cmsLog_error("cmsImg_saveIfNameFromSocket: Error in getsockname.");
      return CMSRET_INTERNAL_ERROR;
   }

   /* cmsLog_error("cmsImg_saveIfNameFromSocket: Session comes from: %s", inet_ntoa(((struct sockaddr_in *)&local_addr)->sin_addr)); */
   
   if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) 
   {
      cmsLog_error("cmsImg_saveIfNameFromSocket: Error openning socket when getting socket intface info");
      return CMSRET_INTERNAL_ERROR;
   }
   
   numifs = 16;

   bufsize = numifs*sizeof(struct ifreq);
   all_ifr = (struct ifreq *)cmsMem_alloc(bufsize, ALLOC_ZEROIZE);
   if (all_ifr == NULL) 
   {
      cmsLog_error("cmsImg_saveIfNameFromSocket: out of memory");
      close(fd);
      return CMSRET_INTERNAL_ERROR;
   }

   ifc.ifc_len = bufsize;
   ifc.ifc_buf = (char *)all_ifr;
   if (ioctl(fd, SIOCGIFCONF, &ifc) < 0) 
   {
      cmsLog_error("cmsImg_saveIfNameFromSocket: Error getting interfaces\n");
      close(fd);
      cmsMem_free(all_ifr);
      return CMSRET_INTERNAL_ERROR;
   }

   numifs = ifc.ifc_len/sizeof(struct ifreq);
   /* cmsLog_error("cmsImg_saveIfNameFromSocket: numifs=%d\n",numifs); */
   for (i = 0; i < numifs; i ++) 
   {
	  /* cmsLog_error("cmsImg_saveIfNameFromSocket: intface name=%s\n", all_ifr[i].ifr_name); */
	  struct in_addr addr1,addr2;
	  addr1 = ((struct sockaddr_in *)&(local_addr))->sin_addr;
	  addr2 = ((struct sockaddr_in *)&(all_ifr[i].ifr_addr))->sin_addr;
	  if (addr1.s_addr == addr2.s_addr) 
	  {
	      strcpy(connIfName, all_ifr[i].ifr_name);
	  	break;
	  }
   }

   close(fd);
   cmsMem_free(all_ifr);

   cmsLog_debug("connIfName=%s", connIfName);

   return CMSRET_SUCCESS;
   
}
